/** $(c2)
 *
 * Copyright (C) 2013-2024 David Azarewicz <david@88watts.net>
 * All rights reserved.
 *
 * LICENSE
 *
 * You may copy this source code, use it in derivative works, or use it in any
 * other way, provided the copyright notice above is not removed.
 *
 * This source code is provided to you with absolutely no warranty whatsoever.
 * There is no guarantee that you will be able to use this code, or build the
 * program, or use it in any way. It is provided solely as a reference so that
 * you may understand how the program works.
 *
 * The author is not responsible in any way for any problem caused by this
 * software, whether it is used correctly or not.
 */
#include "Dev32lib.h"

/* This is the name of your driver. It is here so that we can use it
 * multiple times for displaying messages, etc.
 */
char cDevName[] = "MYDRV$";

void StrategyInit(REQPACKET *prp)
{
  short Success;

  /* You must call this function to put your driver name into the device
   * driver header. The UtSetDriverName() function ensures that the driver
   * name complies with all the requirements (max 8 chars, blank padded).
   */
  UtSetDriverName(cDevName);

  Success = 1;

  if (Success)
  {
    /* You must return a status of RPDONE */
    prp->usStatus = RPDONE;

    /* You can optionally output a message to the screen */
    iprintf("Driver %s loaded successfully.\n", cDevName);
  }
  else
  {
    /* Return done and error */
    prp->usStatus = RPDONE | RPERR;

    iprintf("Driver initialization failed. %s not loaded.\n", cDevName);
  }
}

/* If you want an IDC routine it should be defined this way:
 *  void IdcHandler(REQPACKET *prp)
 *  {
 *  }
 * If you do not define an IDC handler, a default one that does nothing
 * is automatically linked in.
 */

/* You must have a StrategyHandler routine and it must be defined this
 * way. The kernel calls this routine whenever it wants to access the
 * driver
 *
 * An OS/2 driver should have the folowing strategy commands defined
 * as shown below, even if they do nothing.
 *
 * In a 32 bit driver all strategy commands are called at ring 0.
 */
void StrategyHandler(REQPACKET *prp)
{
  /* This is the default status returned unless changed by a function
   * below. You must always at least have the RPDONE bit set upon
   * return even if you also have other bits set as well.
   */
  prp->usStatus = RPDONE;

  switch (prp->bCommand) {

  case STRATEGY_INIT:
    /* The STRATEGY_INIT command is issued by the kernel immediately
     * after your driver is loaded. This is so that it can initialize
     * itself and report back to the kernel if it can run or not. There
     * are some minimum things you must do in the init routine. See the
     * StrategyInit() function.
     */
    StrategyInit(prp);
    break;

  case STRATEGY_OPEN:
     /* You only need to implement the OPEN command if you do file
      * open/close functions.
      */
     break;

  case STRATEGY_CLOSE:
     /* You only need to implement the CLOSE command if you do file
      * open/close functions.
      */
     break;

  case STRATEGY_GENIOCTL:
     /* You only need to implement the IOCTL command if you want to
      * accept IOCTL calls from applications.
      */
    prp->usStatus = RPDONE | RPERR | RPERR_BADCOMMAND;
    break;

  case STRATEGY_DEINSTALL:
     /* This command is not needed. */
    break;

  case STRATEGY_INITCOMPLETE:
    /* The INITCOMPLETE command is called after all of the drivers in
     * the system have been loaded. This is to allow your driver to
     * complete any initialization in needs to. If you need to do IDC
     * calls to other drivers, you can be sure all driver are present
     * at this time.
     */
    break;

  case STRATEGY_SAVERESTORE:
    /* you only need SAVERESTORE if you support suspend / resume. */
    break;

  default:
    /* This returns a failure status for all other commands */
    prp->usStatus = RPDONE | RPERR | RPERR_GENERAL;
    break;
  }
}

