/* #! /usr/bin/regina */
/* Enhanced Directory Tree Utility by Al Heath */

/* UNDOCUMENTED /DBG option:
    /DBG or /DBG1 = Pause after mask breakdown and starting directory information
    /DBG2 = display intermediary 'listDirectory' results
    /DBG3 = turn on trace("?R") to allow setting 'gblTrace' to part of a filename
            for more debugging of 'filterAllFiles'.                                */

versionString = '2.0.9b  2025-12-27'

/* define some default values */
numeric digits 16        /* increase the precision for byte totals */
pauseOpt = 0             /* a '/p' option */
linesOutput = 1          /*  for use with the /p option */
maxRecursionLevel = 200  /* in practicality, will never have this many nested subdirectories */
dbg = 0                  /* internally debugging, 0 = production no debug mode */
fsSeparator = '\'        /* typical OS/2 or Windows path separator vs Linux/Unix/Aix */

parse version callingEnv
ReginaRexx = pos('REGINA',translate(callingEnv))    /* test if running under ReginaRexx */
parse source opSys . ourFullName .
parse value filespec('name',ourFullName) with exname '.' exExtension
exExtension = '.'||exExtension    /* implied naming convention for Rexx Execs */

if OpSys = 'UNIX' then fsSeparator = '/'

if ReginaRexx > 0 then do                       /* OS/2 vs ReginaRexx | ooRexx */
   Call RxFuncAdd 'SysLoadFuncs', 'RegUtil', 'SysLoadFuncs'
   Call SysLoadFuncs
                                                /* Initialize RxUtils Functions */
   Call RxFuncAdd 'RxLoadFuncs', 'RXUTILS', 'RxLoadFuncs'
   Call RxLoadFuncs('QUIET')
end
else if translate(left(OpSys,7)) = 'WINDOWS' | translate(left(OpSys,4)) = 'OS/2' then do
   /* Insure the SysXXXXX functions are ready to use */
   Call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
   Call SysLoadFuncs
end

/* Read screen size to implement paging the output if requested by '/p' */
parse value SysTextScreenSize() with rows cols

/* Special file name to signal 'abort further execution' */
abortSignalFileName = left(ourFullName,lastpos('.',ourFullName))||'STP'

parse arg args
if strip(translate(args)) = '!STOP' then do
   rc = lineout(abortSignalFileName,'Interrupt processing at' date() time())
   rc = stream(abortSignalFileName,'c','close')
end

if stream(abortSignalFileName,'c','query exists')  <> '' then do
   say 'abort request detected as "'||abortSignalFileName||'" exists!'
   '@pause'
   rc = SysFileDelete(abortSignalFileName)
   exit 2
end

/* Parsing of masks is by '+' however the operator may think '&' is also a
   natural clause separator, so we insert the '+' for them if they forgot it. */
i = pos('&',args)
do while i > 0
   if substr(args,i-1,1) <> '+' then do
      args = left(args,i-1)||'+'||substr(args,i)
      i = i + 1
   end
   i = pos('&',args,i+1)
end

if word(args,1) = '?' | word(args,1) = '-?' | translate(word(args,1)) = '-H' then do
   'cls'
   pos = SysCurPos(rows-20,0)
   say 'Enhanced directory command for a drive, vaguely similar to "dir /s".'
   say '    ' filespec('name',ourFullName) 'Version:' versionString
   say '  A file mask may be specified.  Defaults to *  (Be CAREFUL, * is greedy!)'
   say '  Implied path of ".\" to start from the current relative subdirectory.'
   say '  Quote the "mask" with Single or Double quotes when it has embedded blanks.'
   say '  The file mask processing is greatly enhanced from the standard "dir" command.'
   say ''
   say 'Syntax of "'||ourFullName||'":'
   say ' ' filespec('name',exname) '<mask> <modifiers> ( <DIR | FILES | BOTH> <COUNT | ROLLUP>'
   say ''
   say 'Hint: Abort another instance when "'||abortSignalFileName||'" exists!'
   say '  By running "'||ourFullName '!STOP"'
   say ''
   say '  Use the /VER option to only display this exec version information.'

      parse value SysCurPos() with cursorRow cursorCol
      rc = charout(,'Press ENTER to continue...Anything else to quit.')
      pos = SysCurPos(cursorRow,6)
      pull response
      if response <> '' then exit
      pos = SysCurPos(cursorRow-1,0)
      say copies(' . ',trunc((cols-3)/3))

   say ''
   say 'Optionally, The <mask> may be have multi fragments WITHIN double quotes.'
   say '  NOTE: EACH MASK PART IS SEPARATED BY A "+" SYMBOL. See the "&" example below.'
   say '  The <mask> is NOT a regular expression.  A leading caret "!" means NOT.'
   say 'For Example:'
   say '  "*.dll+*.cmd"  will return all *.dll PLUS all *.cmd matches.'
   say '  "!*.cmd"  will return all that are NOT *.cmd.'
   say '  A fragment within parenthesis means the subdirectory must contain "mask".'
   say '     "(*.jpg)" ignore the ENTIRE subdirectory unless it contains a *.jpg file.'
   say '  "*.htm*+(*.jpg)" is *.htm* files in subdirectories with 1 or more jpg files.'
   say '  "*.htm*+(!*.jpg)" is *.htm* files in subdirectories without a jpg file.'
   say ''
   say ' NOTE: "e*+!*.cmd" returns "e*" plus "!*.cmd", thus "e.cmd" would be returned'
   say '    as it matched the "e*" fragment. Likewise "a.dll" would also be returned!'
   say '    That may not be what you were thinking it might do!'
   say '    You might want an "&" AND condition: Use "e*+&!*.cmd"   QUOTES REQUIRED!'
   say '  An fragment starting with an "=" sign filters the current result to matching'
   say '     names in another location.  It may be combined with != to indicate NOT.'
   say '     In these "=" cases, only the matching names are used in the filter.'
   say '     For example "\*.dll+&=\ecs\dll\*.dll" to find any DLLs also in \ecs\dll'
   say ' Try "/E" to Explain how specific masks will function.'
   if rows < 42 then do
      parse value SysCurPos() with cursorRow cursorCol
      rc = charout(,'Press ENTER to continue...Anything else to quit.')
      pos = SysCurPos(cursorRow,6)
      pull response
      if response <> '' then exit
      pos = SysCurPos(cursorRow-1,0)
      say copies(' . ',trunc((cols-3)/3))
   end
   else say ''

   say 'Options:'
   say '  Default is to display only files in ALL subdirectories.'
   say '  Option DIR means only the directory tree in ALL subdirectories.'
   say '    **NOTE: using "/a:d" will also display the directory tree.'
   say '  Option FILES means only Files in the ONE SPECIFIED subdirectory.'
   say '    **NOTE: The FILES option is deprecated.  Use "/r0" instead.'
   say '  Option BOTH displays all files and all directory names.'
   say ''
   say '  Option COUNT will output only summary information of # matches and bytes'
   say '       used by each directory.'
   say '    A minimum COUNT may be specified, i.e. "COUNT 2", to display only the'
   say '       entries that have 2 or more mask matches, or "COUNT 0" for 0 or more.'
   say '       Default is 1 or more matches.'
   say '  Option ROLLUP is same as COUNT with the addition of lower level subdirectory'
   say '    summary count information is rolled up into the parent directory.'

   parse value SysCurPos() with cursorRow cursorCol
   rc = charout(,'Press ENTER to continue...Anything else to quit.')
   pos = SysCurPos(cursorRow,6)
   pull response
   if response <> '' then exit
   pos = SysCurPos(cursorRow-1,0)
   say copies(' . ',trunc((cols-3)/3))

   say 'Modifiers may be:'
   say ' Comparators:'
   say '   /d: signals a date comparison for greater than or equal to specified date.'
   say '            Note: date format is mm-dd-yy.  mm/dd/yy is not allowed.'
   say '            Year may be 2 digit "yy" or for fully "yyyy".'
   say '        if /d: is given without a date, defaults to "equals today".'
   say '        optionally, /de: for date equal to,'
   say '                 or /dl: for date less than or equal.'
   say '                 or /dg: for date greater than or equal.'
   say '        optionally a second date modifier may be given to specify a date range.'
   say '             i.e. /dg:2-13-24 /dl:4-15-24  for an inclusive date range.'

   if rows < 25 then do
      parse value SysCurPos() with cursorRow cursorCol
      rc = charout(,'Press ENTER to continue...Anything else to quit.')
      pos = SysCurPos(cursorRow,6)
      pull response
      if response <> '' then exit
      pos = SysCurPos(cursorRow-1,0)
      say copies(' . ',trunc((cols-3)/3))
   end

   say '   /t: signals a time comparison for greater than or equal to specified time.'
   say '        if /t: given without a date, the date defaults to today.'
   say '     Typically, if a /dg: is specified, specify only /t: to select those'
   say '           files newer than that date and time.  If a /tg: is ALSO specified,'
   say '           then BOTH the Date and the Time must be greater FOR EACH DAY!.'
   say '        *** IMPORTANT NOTE ***  That probably would not be what you intended!'
   say '        BE CAREFUL IF YOU USE: /te: for time equal to,'
   say '                 or /tl: for time less than or equal.'
   say '                 or /tg: for time greater than or equal.'
   say '   /s: signals a size comparison.  (defaults to /sg:1  >=1 or not zero bytes).'
   say '        Similarly there are the /sl: and /se: options available.'
   say '        Standard (1024) suffixes for K,M, or G (Kilo, Mega, Giga) bytes.'
   say '   /a: attributes must have.  For example /a:a-r for Archive AND not readonly'
   say '                                  or /a:hs for hidden AND system files.'

   parse value SysCurPos() with cursorRow cursorCol
   rc = charout(,'Press ENTER to continue...Anything else to quit.')
   pos = SysCurPos(cursorRow,6)
   pull response
   if response <> '' then exit
   pos = SysCurPos(cursorRow-1,0)
   say copies(' . ',trunc((cols-3)/3))

   say 'Output control:'
   say '  /l /l+ display the .LONGNAME extended attribute as the file name'
   say '       instead of the actual short name on a VFAT file system.'
   say '  /b  bare means no date, size, attributes.'
   say '  /fn filename only portion instead of the fully qualified path.'
   say '  /EA include size of EAs as 4th word in the output.'
   say '       /EA+ will only include files with EAs.'
   say '  /p  to page the output considering the screen size.'
   say "      In /exec mode, pause after each exec'd command so output can be checked."
   say "         Also only for /exec mode, /p<!><value>"
   say "                 to pause only when the exec'd RC is = or not = the value."
   say '  /r# maximum subdirectory recursion level.'
   say '      Default is unlimited.  Specify "/r0" for no recursion.'
   say "  Order (within directory) by '/o:n | /o:d | /o:s' for Name or Date or Size."
   say "      sort modifier 'R' for Reverse sort.  i.e. /o:r  or /o:sr   etc."
   say "      '/o:' or '/o' defaults to sorting by name."
   say "     Very Unusal, but you can sort by TIME ONLY with /o:t vs /o:d date & time"

   parse value SysCurPos() with cursorRow cursorCol
   rc = charout(,'Press ENTER to continue...Anything else to quit.')
   pos = SysCurPos(cursorRow,6)
   pull response
   if response <> '' then exit
   pos = SysCurPos(cursorRow-1,0)
   say copies(' . ',trunc((cols-3)/3))

   say '  /cmd xxx'
   say '      also implies the /b (bare) modifier.'
   say '      xxx is a command to prepend to the fully qualified name.'
   say '       OR a command template where % indicates where to substitute the name.'
   say '         % is full drive:\path\name, %D Drive only, %P Path only, %N Name only.'
   say '                for a path relative to the starting location, use %R.'
   say '                for path without a trailing "'||fsSeparator||'", use %p     (small p).'
   say '                for name without a file extension, use %n   (small n).'
   say '    IMPORTANT: quote the entire xxx string when the template is multiple words.'
   say '         if the string embeds both "'||" and '"
   say '               then use a backtick "`" to define the string.' 
   say '               OR double escape your embedded single quotes.'
   say '     Typically one would redirect this output to a file to further manipulate.'
   say '  /exec xxx is similiar to /cmd except it will execute it immediately!'
   say ''
   if  translate(left(OpSys,4)) = 'OS/2' then do
      say 'Enter "G" to Explain Sample masks, or any other key to exit Help ...'
      pull response
      if translate(left(response,1)) <> 'G'
         then exit 100
   end
   else do
      '@pause'
      exit 100
   end

   /* our examples should work with 'boot drive' & '\os2' */
   Call RxFuncAdd 'RxLoadFuncs', 'RXUTILS', 'RxLoadFuncs'
   Call RxLoadFuncs('QUIET')
   forceDrive = filespec('drive',directory())
   forceDir = filespec('path',directory()||fsSeparator)
   if translate(forceDrive) <> RxBootDrive()
      then forceDrive = RxBootDrive()
      else forceDrive = ''
   if translate(forceDir) <> '\OS2\'
      then forceDir = '\OS2\'
      else forceDir = ''
   ExampleArgs.1 = '"'||forceDrive||'\os2\*.ini" Simple example that forces the starting path as "\os2" and traverses that tree looking for "*.ini".',
                                       '  If the current directory was already "\os2" then it is equivalent to a simple mask of just "*.ini".'
   ExampleArgs.2 = '"'||forceDrive||forceDir||'boot\*.ini+\os2\*.ini" Notice how this complex mask operates differently from "'||RxBootDrive()||forceDir||'*.ini" ',
                                       'since the 2nd mask is fixed from the drive root.  With the first mask specifying the starting directory ',
                                       'as "\os2\boot", the search will NOT find ini files in "\os2\install" nor "\mdos\winos2"'
   ExampleArgs.3 = '"'||forceDrive||'\ecs\*.ini+..\os2\os2.ini" This one is tricky to understand since the ..\os2 is applied as the \ecs tree is traversed.',
                                       '  Thus when in \ecs it will look at \os2 but then in \ecs\boot it will look for \ecs\os2 which likely does not exist.'
   ExampleArgs.4 = '"'||forceDrive||forceDir||'*inst\*.ini" Only those in subdirectories with an immediate path matching "*inst".'
   if forceDir = ''
      Then exampleArgs.4 = '"'||forceDrive||'.\*inst\*.ini" Only those in subdirectories with a path matching "*inst". ' ,
                                        'NOTE: In this case, the ".\" is VERY important to make it a relative in the tree!'
   ExampleArgs.5 = '"'||forceDrive||forceDir||'*.ini+(!shut*.exe)" In this case, the subdirectory MUST ALSO NOT CONTAIN 1 or more shut*.exe files. ',
                                       'A "contains" clause, denoted by ( ), is a simple true/false logic test to pass before continuing.'
   ExampleArgs.6 = '"'||forceDrive||forceDir||'*.ir+.\*inst\*ini" Demonstrates the importance of ".\" in the 2nd specification.' ,
                                       'Without the ".\", only *inst subdirectories directly under \os2 are searched.'
   ExampleArgs.7 = '"'||forceDrive||forceDir||'*.ini+(*.exe)+(!write.exe)" Returns INI files in directories that contain *.exe but NOT write.exe' ,
                                       'In this extremely concocted case, ',
                                       'excluding those MDOS\WINOS2\*.INI files as WRITE.EXE exists there, but still includes MDOS\WINOS2\SYSTEM\*.ini ',
                                       'files as SYSTEM has an EXE but not WRITE.EXE.'
   ExampleArgs.8 = '"'||forceDrive||forceDir||'*.ini+&!atm.ini+&!.\ARCHIVES\*+&!install\*" Returns INI files except any ATM.INI and excludes recursing ',
                                       'ARCHIVES and INSTALL subdirectories.  Note significance of "&" meaning an AND condition.'
   ExampleArgs.9 = '"'||forceDrive||'\*.dll+&=\ecs\dll\*.dll" Returns any DLL name that is also present in the \ecs\dll subdirectory '
   ExampleArgs.10 = '"'||forceDrive||'\usr\lib\*.dll+&!=\ecs\dll\*.dll" Another example of a NOT condition: Returns any \usr\lib DLL name that ',
                                       'is NOT present in the \ecs\dll subdirectory '

   ExampleArgs.0 = 10

   say ''
   if forceDrive <> '' | forceDir <> '' then do
      say 'These specific examples may over specify the starting directory so they'
      say '   will actually run regardless of your actual current drive and directory.'
      say '   In practice, you will likely run the exec assuming the current drive'
      say '   and probably the current working directory.'
      say ''
   end
   else do
      say 'Changing your default drive and directory will affect how these examples'
      say 'are defined! They have been designed to search the OS2 directory on your'
      say 'boot drive even when this exec has been invoked from a different location.'
      say 'Only the example masks will change but the result will be consistent.'
   end
   say 'Note, Your current directory is:' substr(directory(),3) 'and thus is implied!'

   l = 6+2
   do i = 1 to ExampleArgs.0
      parse value ExampleArgs.i with '"' Sample '"' Text
      say i '"'||word(Sample,1)||'"'
      l = l + 1
      if Text <> '' then do until Text = ''
         if l >= rows then do
            parse value SysCurPos() with cursorRow cursorCol
            rc = charout(,'Press ENTER for more...')
            pos = SysCurPos(cursorRow,6)
            pull response
            pos = SysCurPos(cursorRow-1,0)
            l = 2
         end
         j = ''
         do until length(j) + 1 + length(word(Text,1)) > cols - 10
            j = j word(Text,1)
            Text = subword(Text,2)
            if Text = '' then leave
         end
         say '    ' j
         l = l + 1
      end
   end
   say ''

   say 'Select an example, or just press Enter to exit this help section...'
   say '    (Remember you can always use the "/E" modifier to Explain a mask.)'
   parse pull response
   if response = '' | datatype(response) = 'NUM'
      then if response < 0 | response > ExampleArgs.0
         then exit 100
   if datatype(response) = 'NUM'
      then parse value ExampleArgs.response with '"' args '"' .
      else args = response
   args = args '/E'                   /* we always want to Explain this run */
   parse arg . '/' i '(' .      /* include any additional args we were passed */
   if i <> ''
      then args = args '/'||i
end

/* take an early check for a /DBG option to help us debug parsing */
if pos('/DBG',translate(args)) > 0 then do     /* 'Debug' */
   dbg = 1
   i = pos('/DBG',translate(args))+4
   if substr(args,i,1) <> ' ' & datatype(word(substr(args,i),1)) = 'NUM'
      then dbg = word(substr(args,i),1)
end

if pos('/VER',translate(args)) > 0 then do     /* 'Version' */
   say '    ' filespec('name',ourFullName) 'Version:' versionString
   exit 1
end
/* first lets pull the mask specification from the args */
mask = ''
do w = 1 to words(args)
   /* find the first word that isn't a modifier option */
   if left(word(args,w),1) <> '/' then do

      /* it could have be single or double quoted string to treat as one entity */
      ch = left(word(args,w),1)        /* if could have been single or double quoted */

      /* if this is part of an /exec or /cmd string */
      if w >= 2
         then if ((translate(word(args,w-1)) = '/EXEC') | (translate(word(args,w-1)) = '/CMD')) then do

         if (ch = '"' | ch = "'" | ch = '`') then do
            strA = subword(args,w)        /* pull out the start of the quoted string */
            i = pos(strA,args)            /* where the string starts in the original args */
            j = pos(ch,substr(strA,2))    /* the end of the actual quoted string */
            if j=0 then do
               say 'Unpaired quote' ch 'in parameter string.  Abort processing.'
               exit 8
            end
            strB = left(strA,j+1)

            /* the ending quote must not be further embedded itself */
            if length(strA) > length(strB) & substr(strA,length(strB)+1,1) <> ' ' then do
               say 'Expecting a blank after embedded quoted string' strB
               say '  Parameter string:' args
               exit 8
            end

            /* skip over then entire quoted string as we continue looking for the mask */
            w = w + words(strB)-1
         end
         iterate
      end

      /* when no mask no modifier options, just the old style '( opts' */
      if ch = '(' & 0 = pos(')',substr(subword(args,w),2)) then leave

      if (ch = '"' | ch = "'" | ch = '`') then do
         mask = subword(args,w)        /* pull out the start of the mask */
         i = pos(mask,args)            /* where the mask starts in the original args */
         j = pos(ch,substr(mask,2))    /* the end of the actual quoted mask */
         if j > 0 then do
            mask = substr(mask,2,j-1)  /* limit the mask to the ending quote */
            args = strip(left(args,i-1)) strip(substr(args,i+length(mask)+2))  /* remove mask from the arg string */
         end
         else do
            say 'Error parsing the mask.  Looks like it is missing a ending quote:' ch
            say '    ' args
            exit 8
         end
      end
      else do
         mask = word(args,w)
         args = delword(args,w,1)
      end

      /* insure the mask is cleaned up */
      mask = strip(strip(mask),'B','"')
      leave
   end
end
If mask = '' | mask = '.' Then mask = '*'
maskArg = mask

/* now lets make the initial separation of the options from the modifier args */
i = wordpos('/CMD',translate(args))
if i = 0
   then i = wordpos('/EXEC',translate(args))
if i > 0 then do
   /* handle an embedded '(' within a CMD or EXEC template */
   ch = left(word(args,i+1),1)
   if (ch = '"' | ch = "'" | ch = '`') then do
      sz = subword(args,i+1)
      c = pos(ch,substr(sz,2))
      if c > 0 then do
         j = pos('(',sz,c)
         if j > 0 then do
            opts = substr(sz,j+1)
            args = subword(args,1,i) substr(sz,1,j-1)
         end
         else opts = ''     /* there is no '(' to worry about */
      end
      else do
         say 'Error parsing the' word(args,i) 'template.  Looks like it is missing a ending quote:' ch
         say '   ' sz
         exit 8
      end
   end
   else do
      /* the template wasn't quoted so it is also a simple parse */
      parse arg . '/' args '(' opts
      args = '/'||strip(args)
   end
end
else do   /* else nothing special to consider when parsing */
   parse var args . '(' opts
   parse var args . '/' args '(' .
   if strip(args) <> ''
      then args = '/'||strip(args)
end
opts = translate(opts)        /* we expect upper case */

/* Debug: dump out our parsing results */
if dbg > 0 then do
   parse arg i
   say 'Parsing:' i
   say '  Mask: "'||mask||'"'
   say '  Modifiers:' args
   say '  Options:' opts
end

/* handle some old options that may be deprecated */
what = 'F'      /* only files listed by default */
If wordpos('DIR',translate(strip(opts))) > 0 Then what = 'D'
If wordpos('BOTH',translate(strip(opts))) > 0 Then what = 'B'

nesting = 1   /* assume we will traverse all nested subdirectories */
If wordpos('FILES',translate(strip(opts))) > 0 Then nesting = 0

countOnly = -1 /* by default, names will be displayed */
rollup = 0     /* by default, counts are not rolled up to parent directory */
i = wordpos('COUNT',translate(strip(opts)))
If i = 0 then do
   i = wordpos('ROLLUP',translate(strip(opts)))
   If i > 0
      then rollup = 1
End
If i > 0 Then Do
   countOnly = 1
   if words(opts) > i & datatype(word(opts,i+1)) = 'NUM'
      then countOnly = word(opts,i+1)
end

/* set special processing options from the modifier args */
if pos('/DBG',translate(args)) > 0 then do     /* 'Debug' */
   dbg = 1
   i = pos('/DBG',translate(args))+4
   if substr(args,i,1) <> ' ' & datatype(word(substr(args,i),1)) = 'NUM'
      then dbg = word(substr(args,i),1)
end

pauseOpt = pos('/P',translate(args))
if pauseOpt > 0 then do                      /* 'Pause' arg */
   execPauseRC = substr(args,pauseOpt)             /* is there a specific value to check for on /EXEC results */
   execPauseRC = substr(word(execPauseRC,1),3)
   pauseOpt = 1
end
else execPauseRC = ''

if pos('/R',translate(args)) > 0 then do     /* 'Recursion' arg */
   i = pos('/R',translate(args))+2
   if substr(args,i,1) <> ' ' & datatype(word(substr(args,i),1)) = 'NUM'
      then maxRecursionLevel = word(substr(args,i),1)
end

show_longname = 0
i = pos('/L',translate(args))
if i > 0 then do
   parse value substr(args,i) with _OptValue .
   show_longname = 1
   if length(_OptValue > 2) & right(_OptValue,1) = '+'
      then show_longname = 2
   args = delstr(args,i,length(_OptValue))
end

showFullNameOnly = 0
if pos('/B',translate(args)) > 0
   then showFullNameOnly = 1       /* don't display date/time, size, attributes */

bIncludeEA = 0           /* don't include EA size information as the 4th value (just after file size) */
if pos('/EA',translate(args)) > 0
   then bIncludeEA = 1                  /* include EA size information */
if pos('/EA+',translate(args)) > 0
   then bIncludeEA = 2                  /* only files with EAs */

showFullPath = 1
if pos('/FN',translate(args)) > 0
   then showFullPath = 0           /* don't display the drive/path as part of the name */

execTemplate = ''
i = wordpos('/CMD',translate(args))
if i = 0
   then i = wordpos('/EXEC',translate(args))
if i > 0 then if words(args) > i then do
   showFullNameOnly = 2
   if wordpos('/EXEC',translate(args)) > 0
      then showFullNameOnly = 3
   execTemplate = word(args,i+1)
   ch = left(execTemplate,1)
   if ch = '"' | ch = "'" | ch = '`' then do
      execTemplate = substr(subword(args,i+1),2)
      c = pos(ch,execTemplate)
      execTemplate = left(execTemplate,c-1)
   end
   args = delword(args,i+1,words(execTemplate))
   if pos('%',execTemplate) = 0
      then execTemplate = execTemplate '%'
end

/* date comparisons */
end_date = 0
if pos('/d:',args) > 0 | pos('/de:',args) > 0 | pos('/dg:',args) > 0 | pos('/dl:',args) > 0 then do
   parse var args . '/d' comparator ':' begin_date . '/' nextArgs

   /* Is there a second date argument to specify a range of dates? */
   if nextArgs <> '' then do
      /* set the end_date value IF it was also specified */
      nextArgs = translate('/'||nextArgs )
      parse var nextArgs . '/D' d_comparator2 ':' end_date . '/' .
      if d_comparator2 <> '' & (translate(comparator) <> 'G' | d_comparator2 <> 'L') then do
         say 'Incorrect date range specification.'
         say '  the first date must be "G" for Greater than or equal'
         say '  and the second date must be "L" for Less than or equal to.'
         exit 8
      end
      if pos('-',end_date) > 0
         then parse var end_date mm '-' dd '-' yy
         else parse var end_date mm '/' dd '/' yy

      if end_date <> '' then do
         if yy = '' then yy = left(date('S'),4)
         if yy <= 99 & yy >= 70
            then yy = yy + 1900
            else if yy < 100 then yy = yy + 2000

         if (( mm < 1 | mm > 12) | (dd < 1 | dd > 31)) then do
            say '  Ending date entered was not in proper month-day-year form:'
            say '     a date of' end_date 'is not appropriate.'
            exit 8
         end

         end_date = yy||'-'||right(100+mm,2)||'-'||right(100+dd,2)
      end
      else end_date = 0
   end

   comparator = translate(comparator)
   if begin_date = '' then do
      begin_date = date('U')

      if dbg > 0
        then say '   defaulting date to today =' begin_date
   end

   if pos('-',begin_date) > 0
      then parse var begin_date mm '-' dd '-' yy
      else parse var begin_date mm '/' dd '/' yy

   if yy = '' then yy = left(date('S'),4)
   if yy <= 99 & yy >= 70
      then yy = yy + 1900
      else if yy < 100 then yy = yy + 2000

   if (( mm < 1 | mm > 12) | (dd < 1 | dd > 31)) then do
      say '  Date entered was not in proper month-day-year form:'
      say '     a date of' begin_date 'is not appropriate.'
      exit 8
   end

   begin_date = yy||'-'||right(100+mm,2)||'-'||right(100+dd,2)
end
else do
  begin_date = 0
  comparator = ''
end

/* time comparisons */
parse var args . '/t' t_comparator ':' time_arg . '/' nextArgs
t_comparator = translate(t_comparator)
if time_arg = ''
   then time_arg = '00:00'
else if nextArgs <> ''
   then nextArgs = translate('/'||nextArgs )

pm = 0
t_precision = 'MINUTES'              /* assume timespec is Hours & Minutes */
parse upper var time_arg hr ':' min ':' sec .
if min = '' then do
   min = 0                           /* specified time is only the hours */
   sec = 0
   t_precision = 'HOURS'
   if right(hr,1) = 'P' then do
      pm = 12
      hr = left(hr,length(hr)-1)
   end
   else if hr = '12A' then do
      hr = 0
   end
end
if right(min,1) = 'P' then do
   if hr <> 12
      then pm = 12
   min = left(min,length(min)-1)
end
else if right(min,1) = 'A' then do
   min = left(min,length(min)-1)
   if hr = 12
      then pm = -12
end
if sec = '' then sec = 0
else do
   t_precision = 'SECONDS'
   if right(sec,1) = 'P'  then do
      if hr <> 12
         then pm = 12
      sec = left(sec,length(sec)-1)
   end
   else if right(sec,1) = 'A' then do
      sec = left(sec,length(sec)-1)
      if hr = 12
         then pm = -12
   end
end
begin_time = (((hr + pm) * 60) + min) * 60 + sec

implied_end_date = ''
if begin_time > 0 & begin_date = 0 then do
   begin_date = substr(date('S'),1,4)||'-'||substr(date('S'),5,2)||'-'||substr(date('S'),7,2)
   implied_end_date = begin_date       /* just in case they wanted between times for a default of today */
end

t_comparator2 = ''
end_time = 0
if nextArgs = '' then do
   /* there is no ending time specification but if there was a beginning time and end date
      then we assume then end of day. */
   if end_date > 0 & begin_time > 0 then do
      /* assume they mean any time in the day greater than begin_time */
      end_time = (24 * 60 * 60) + 1
      if t_comparator <> '' then do
         /* unless we are getting into some really wierd requested time specifications */
         if t_comparator = 'G' then t_comparator2 = 'L'
         else if t_comparator = 'L' then t_comparator2 = 'G'
         else if t_comparator = 'E' then do
            t_comparator2 = 'E'
            end_time = begin_time
         end
      end
   end
end
else do
   /* set the end_time value IF it was also specified */
   parse upper var nextArgs . '/T' t_comparator2 ':' end_time . '/' .
   if end_time = '' then do
      end_time = 0
   end
   else do     
      if implied_end_date <> ''
         then end_date = implied_end_date
      if end_date <> begin_date & (t_comparator2 <> '' | t_comparator <> '') then do
         say 'Possible incorrect time range specification.'
         say '  Did you really mean to limit each day to those specific times?'
         say '  If you really intended a full date range, do not specify /T'||t_comparator||': and /T'||t_comparator2||':'
         say '     just specify the times with the /T: option.'
         if t_comparator = '' | t_comparator = '' then do
            say "I can't figure out what you were intending to specify so will abort."
            exit 8
         end
         say ' Press enter to continue as specified or anything else to abort.'
         pull response
         if response <> ''
            then exit 8
      end

      parse var end_time hr ':' min ':' sec .
      pm = 0
      if min = '' then do
         min = 0
         if right(hr,1) = 'P'  then do
            if hr <> 12
               then pm = 12
            hr = left(hr,length(hr)-1)
         end
         else if right(hr,1) = 'A' then do
            hr = left(hr,length(hr)-1)
            if hr = 12
               then pm = -12
         end
      end
      else do
         /* process times such as 1:15p */
         if right(min,1) = 'P'  then do
            if hr <> 12
               then pm = 12
            min = left(min,length(min)-1)
         end
         else if right(min,1) = 'A' then do
            min = left(min,length(min)-1)
            if hr = 12
               then pm = -12
         end
      end

      if sec = ''
         then sec = 0
      else if right(sec,1) = 'P'  then do
         if hr <> 12
            then pm = 12
         sec = left(sec,length(sec)-1)
      end
      else if right(sec,1) = 'A' then do
         sec = left(sec,length(sec)-1)
         if hr = 12
            then pm = -12
      end

      end_time = (((hr + pm) * 60) + min) * 60 + sec

      if end_date = ''
         then end_date = begin_date
   end
end

/* is there a "size" option? */
s_comparator = ''
if pos('/S',translate(args)) > 0 then do
   parse var args . '/s' s_comparator ':' size_arg . '/' .
   s_comparator = translate(s_comparator)
   if s_comparator = '' then s_comparator = 'G'
   if size_arg = '' then size_arg = 1
   else do
      size_arg = translate(strip(size_arg))
      ch = right(size_arg,1)
      if datatype(ch) <> 'NUM' then do
         /* standard modifiers for size as kilo, mega, giga bytes */
         size_arg = strip(size_arg,'T',ch)
         if ch = 'K' then size_arg = trunc(size_arg * 1024)
         if ch = 'M' then size_arg = trunc(size_arg * 1024 * 1024)
         if ch = 'G' then size_arg = trunc(size_arg * 1024 * 1024 * 1024)
      end
   end
end

/* is there a "ordering" option? */
sortBy = ''
if pos('/O',translate(args)) > 0 then do
   parse value translate(args) with '/O:' sortBy .
   if sortBy = '' then sortBy = 'N'        /* Default to sort by 'name' */
end

tAttrib = "*****"   /* ADHRS attribute mask for SysFileTree */
parse var args . '/a:' attributeFilter .
do while attributeFilter \= ''
   /* parse out the target attribute specification... i.e.  '/a:-r' for not read only */
   anAttr = left(attributeFilter,1)
   anAttrMask = '+'
   if anAttr = '-' then do
      anAttr = left(attributeFilter,2)
      anAttrMask = '-'
   end
   if anAttr = '+' then do
      anAttr = left(attributeFilter,2)
      anAttrMask = '+'
   end
   attributeFilter = substr(attributeFilter,length(anAttr)+1)

   anAttr = translate(right(anAttr,1))
   maskPos = pos(anAttr,'ADHRS')

   tAttrib = overlay(anAttrMask,tAttrib,maskPos,1)
end
if substr(tAttrib,2,1) = '+'
   then what = 'D'

/* On Unix/Linux, Hidden files start with a . */
if OpSys = 'UNIX' & substr(tAttrib,3,1) = '+' & left(mask,1) \= '.' then do
   mask = '.'||mask
   tAttrib = substr(tAttrib,1,2) || '*' || substr(tAttrib,4)
end

/* decompose a complex mask into its subordinate pieces
   ordering "contains" clause(s) first in processing order */
maskParts.0 = 0
fullMask = ''
NotMask = 0
primaryMaskIndex = 0

do  pass = 1 to 2
   parsingMask = mask
   do while parsingMask <> ''
      parse var parsingMask m '+' parsingMask
      if pass = 1 then do      /* first pass is any contains clauses */
         if left(m,1) <> '('
            then iterate
      end
      else if left(m,1) = '('  /* next pass is anything else */
            then iterate

      /* add the mask to the processing list */
      i = maskParts.0 + 1
      maskParts.i = translate(m)
      maskParts.0 = i

      /* the first "non contains" mask could specify a starting path */
      if pass = 2 & fullMask = '' then do
         fullMask = m
         if left(mask,1) = '!' then do
            NotMask = 1
            m = substr(m,2)
         end
         if filespec('drive',m) <> ''
            then m = filespec('path',m)||filespec('name',m)
         primaryMaskIndex = i              /* this is our primary mask */

         /* will the code later on optimize the starting directory? */
         /*   NOTE TO SELF, I think we might need to tweak this a little bit
              as later on we might set the starting dir if there are wild cards
              AFTER some valid starting subdirectories.  For now we won't
              consider the case of 'subdir\*\name' */
         p = filespec('path',m)
         if p <> '' & pos('*',p) = 0 & pos('?',p) = 0 & left(p,2) <> '.'||fsSeparator then do
            maskParts.i = translate(filespec('name',m))
            if NotMask = 1
               then maskParts.i = '!'||maskParts.i
         end
      end

      /* remove any drive specifier in the masks as 'fullMask' has that
         and will cause the proper definition of the root drive.
         Our subsequent processing assumes an active mask has no drive info */
      m = maskParts.i
      j = 1
      if left(m,1) = '('
         then j = j + 1            /* skip over the '(' */
      chAnd = substr(m,j,1)
      if chAnd = '&'               /* remember but skip a '&' */
         then j = j + 1
         else chAnd = ''
      ch = substr(m,j,1)
      if ch <> '!'                 /* remember but skip a '!' */
         then ch = ''
         else j = j + 1
      /* after using scratch variable 'j' as an index we reuse it to reassemble the mask */
      j = strip(substr(m,j),'T',')')
      j = chAnd||ch||filespec('path',j)||filespec('name',j)
      if left(maskParts.i,1) = '('
         then j = '('||j||')'
      maskParts.i = j
   end
end

/* the first 'non contains' clause can specify a starting directory,
   so we strip off any leading '!' that signals a "not" condition */
if left(fullMask,1) = '!' then do
   fullMask = substr(fullMask,2)
end

if dbg > 0 then do i = 1 to maskParts.0
   say 'Masks processed:' i 'as "'||maskParts.i||'"'
end

/* set up for recursion down the directory tree */
gblWarnings = ''
currentRecursionLevel = 0

/* remember where started from in case a different drive was specified */
weAreHere = directory()

/* if the mask doesn't specify a drive */
rootDrive = filespec('drive',fullMask)
if rootDrive = ''
   then rootDrive = filespec('drive',directory())  /* starting a current drive */
rootDrive = translate(rootDrive)
if dbg > 0 then say 'Drive is' rootDrive

/* change to the destination drive */
if translate(filespec('drive',weAreHere)) <> rootDrive
   then '@'||rootDrive

/* remember where we were on the candidate drive */
weWereThere = directory(rootDrive)

/* Now considering the primary mask, see if the drive and/or current directory
   are defaulted. */
relPath = filespec('path',fullMask)
if relPath = '' | (left(relPath,2) = '.'||fsSeparator) then do
   /* no path explicitly specified or it is relative to the current directory,
         so pick up the current directory from the drive */
   '@setlocal'
   relPath = translate(filespec('path',directory(rootDrive)||fsSeparator))   /* add the '\' for path parsing */
   if right(relPath,2) = fsSeparator||fsSeparator                            /* remove it if redundant */
      then relPath = left(relPath,length(relPath)-1)
   '@endlocal'
end
else do         /* else there is a path to consider */

   /* if we could possibly use part of the path to set the starting point of the search,
      however partial paths can't be used to set the initial root directory */

   /* we can't set wild cards in the initial root directory */
   j = pos('*',relPath)
   if j > 0
      then relPath = filespec('path',left(relPath,j-1))
   j = pos('?',relPath)
   if j > 0
      then relPath = filespec('path',left(relPath,j-1))

   /* we'll start at the current directory on the requested drive to fully resolve it */
   '@setlocal'
   d = directory(rootDrive)              /* save where it is */
   j = d                                 /* we don't modify 'd' as it saves our place */
   if right(d,1) <> fsSeparator          /* checking if x:\ or x:\path */
      then j = d||fsSeparator

   if left(relPath,1) = fsSeparator      /* does it specify starting at root */
      then testDir = filespec('drive',j)||relPath||'.'
      else testDir = j||relPath||'.'

   if dbg >= 2 then say 'Verifying "'||testDIr'" exists...'

   parse value translate(directory(testDir)) with . ':' relPath
   i = directory(d)                           /* restore things back */
   '@endlocal'

   /* lets examine the specified path making sure the relevant parts of the path are valid */
   ques = translate(filespec('path',fullMask))
   if ques <> '' then do
      /* ignore generic parent directory specifications */
      do while left(ques,3) = '..'||fsSeparator
         ques = substr(ques,4)
      end
      if left(ques,2) <>'.'||fsSeparator then   /* reserved meanings for '.' in specifying a path */
         itsRelative = 'False'
      else do
         ques = substr(ques,3)
         itsRelative = 'True'
      end

      /* partial paths are not important at this point as we are checking for definite directories */
      j = pos('*',ques)
      if j > 0 then
         ques = filespec('path',left(ques,j-1))
      j = pos('?',ques)
      if j > 0
         then ques = filespec('path',left(ques,j-1))

      /* is a relevant path portion of the mask actually present? */
      if ques <> '' & pos(ques,translate(relPath||fsSeparator)) = 0 then do
         say "Please verify your initial mask path.  It doesn't appear to be valid."
         say '    For:' d '  "'||fullMask||'"'
         exit 8
      end
      else do
         /* it appears to be valid, so remove that part from the mask as it is
            defined by our initial root directory. */
         i = pos(translate(ques),translate(maskParts.primaryMaskIndex))
         if i > 0 & itsRelative = 'False' then do
            /* we make the remaining parts relative and preserve the 'Not' flag */
            maskParts.primaryMaskIndex = substr(maskParts.primaryMaskIndex,i+length(ques))
            if filespec('path',maskParts.primaryMaskIndex) <> ''
               then maskParts.primaryMaskIndex = '.\'||maskParts.primaryMaskIndex
            if NotMask = 1
               then maskParts.primaryMaskIndex = '!'||maskParts.primaryMaskIndex
         end
         if dbg >=2 then say 'relevant path in mask:' ques '& primary mask:' maskParts.primaryMaskIndex
      end
   end
end

rootDir = rootDrive||relPath
/* special case of specifying just a directory without a file mask AND not looking for directories */
if filespec('path',maskParts.1) = '' & pos('*',maskParts.1) = 0 & pos('?',maskParts.1) = 0 & substr(tAttrib,2,1) <> '+' then do
   rc = SysFileTree(rootDir||fsSeparator||maskParts.1,dirs,'DL',tAttrib)
   if dirs.0 = 1 then do
      rootDir = rootDir||fsSeparator||maskParts.1
      maskParts.1 = '*'
      relPath = ''
   end
end
if dbg > 0 then say 'Initial path is "'||relPath||'" from' rootDir

topLevelDir = rootDir
if wordpos('/E',translate(args)) > 0
   then call ExplainMasks

/* before we rock 'n roll, pause and allow trace to be set when Debugging */
if dbg >= 1 then do
   say 'Ready to begin processing...'
   '@pause'
end
if dbg >= 3 then do
   say 'Option here to MANUALLY set "gblTrace" to part of a file name to trace for debugging "filterAllFiles"'
   trace("?R")
end
gblTrace = ''
if dbg >= 3
   then say 'Global Tracing: "'||gblTrace||'"'

/* process the directory recursively */
if rc = 0 then do
   call listDirectory rootDir
   parse pull thisCount ',' thisSize
end

/* restore the original drive and working directories */
weWereThere = directory(weWereThere)
weAreHere = directory(weAreHere)

if countOnly >= 0 then do
   say '"'||rootDir||'"' thisCount 'entries' prettyNum(thisSize) 'bytes'
   parse value SysDriveInfo(left(rootDir,2)) with thisDrive bFree bAvail driveLabel
   say 'Drive' thisDrive '"'||strip(driveLabel)||'" bytes' prettyNum(bFree) 'free of' prettyNum(bAvail) 'available'
end

if gblWarnings <> ''
   then say gblWarnings

exit

quoteString: procedure
/* returns a properly quoted string (using either double or single quotes escaped as necessary) 
   for the input string. */
parse arg _str

   ch = "'"                            /* assume we can single quote the entire string */
   if pos(ch,_str) = 0 
      then _quotedStr = ch||_str||ch
   else do                             /* Oh-oh, it contains a single quote itself */
      ch = '"'
      if pos(ch,_str) = 0            /* if it doesn't have a double quote internally */
         then _quotedStr = ch||_str||ch     /*  we'll double quote the string */
      else do
         /* we have to deal with both single and double quotes, so escape the internal single quotes */
         ch = "'"
         _quotedStr = _str
         do p = pos(ch,_quotedStr) by 0 while p > 0
            _quotedStr = left(_quotedStr,p)||substr(_quotedStr,p)
            if substr(_quotedStr,p+2,1) = ch      /* was the single quote already escaped? */
               then p = p+1                       /* yes, so we must not over achieve. */
            p = pos(ch,_quotedStr,p+2)            /* advance to the next single quote */
         end
                   
         _quotedStr = ch||_quotedStr||ch
      end
   end

return _quotedStr

prettyNum: procedure
parse arg aNumber .
   prettyNumber = reverse(aNumber)
   do nCommas = 1 while length(prettyNumber) >= 4*nCommas
      prettyNumber = substr(prettyNumber,1,4*(nCommas)-1)||','||substr(prettyNumber,4*(nCommas))
   end
   prettyNumber = reverse(prettyNumber)
return prettyNumber

/*************************
 listDirectory proceedure
*************************/

listDirectory: procedure expose exExtension maskParts. begin_date comparator end_date d_comparator2 relPath bIncludeEA ,
                                begin_time t_comparator t_precision end_time t_comparator2 s_comparator size_arg ,
                                show_longname nesting what tAttrib countOnly rollup showFullNameOnly showFullPath execTemplate fsSeparator OpSys pauseOpt execPauseRC ,
                                abortSignalFileName rows cols linesOutput currentRecursionLevel maxRecursionLevel gblWarnings topLevelDir gblTrace dbg sortBy


parse arg rootDir
rootDir = strip(rootDir,'T',fsSeparator)

/* Check for exceeding the recursive subdirectory nesting */
if currentRecursionLevel > maxRecursionLevel then do
   if pos('Subdirectory recursion level',gblWarnings) = 0 & maxRecursionLevel > 0 & countOnly > 0
      then gblWarnings = gblWarnings 'Subdirectory recursion level' currentRecursionLevel 'reached, ignoring lower level subdirectories.'
   push '0,0'
   return
end
currentRecursionLevel = currentRecursionLevel + 1

/* list the contents this directory */
options = what||'L' /* 'L' returns date as YYYY-MM-DD HH:MM:SS */

thisCount = 0
thisSize = 0
consolidatedFiles.0 = 0
allFiles.0 = 0
dirs.0 = 0

      /********* I believe rootDir having a '*' or '?' has been deprecated E **********/
wCard = pos('*',rootDir) + pos('?',rootDir)
if wCard <> 0 then do
   trace("?R")
   say '**** Why did we end up here??? ***'
   rc = SysFileTree(rootDir,dirs,'D',tAttrib)
   do i = 1 to dirs.0
       aSubDir = subword(dirs.i,5)
       call listDirectory aSubDir||fsSeparator
       parse pull subCount ',' subSize
       thisCount = thisCount + subCount
       thisSize = thisSize + subSize
   end
end
else do
   /* process each mask putting all the individual results into stem: files. */
   drop MatchingDirs.
   drop files.
   do m = 1 to maskParts.0

/*
   trace("?R")
*/
      mask = maskParts.m

      /* parse the special meanings from the mask */
      ContainsClause = 0
      if left(mask,1) = '(' then do
         ContainsClause = 1
         parse var mask . '(' mask ')'
      end

      AndMask = ''
      if left(mask,1) = '&' then do
         AndMask = 'And '
         mask = substr(mask,2)
      end

      NotMask = 0
      if left(mask,1) = '!' then do
         NotMask = 1
         mask = substr(mask,2)
      end

      ExcludeMask = 0
      if left(mask,1) = '=' then do
         ExcludeMask = 1
         mask = substr(mask,2)
      end

      /* Check for conditions we haven't coded for yet */
      if NotMask & pos('*',filespec('path',mask)) + pos('?',filespec('path',mask)) > 0 then do
         say 'Warning:  The code can not handle the wild cards as a "not" in "'||mask||'"'
         push '0,0'
         return
      end

      /* mask syntax and meaning, where '!' inverts the test:
            "mask"            search current 'rootDir' directory
            ".\path\mask"     search current 'rootDIr' with relative path
            "path\mask"       search initial 'path' starting directory
            "\path\mask"      search path from ROOT of the drive
      */
      startingAt = rootDir || fsSeparator
      if filespec('path',mask) <> '' then do
         /* we must consider how the 'path' affects the searching */
         if left(mask,3) = '..\' then do
            /* traverse back up the tree to the parent */
            do while left(mask,3) = '..\'
               mask = substr(mask,4)
               startingAt = filespec('drive',startingAt)||filespec('path',strip(startingAt,'T',fsSeparator))
            end
         end
         else if left(mask,1) = '.' then do
            if ContainsClause = 0
            /* it is relative to where we are presently in the directory tree,
                   so we remove the '.\' which means the current directory. */
               then mask = substr(mask,pos(fsSeparator,mask)+1)
               else nop;   /* it will be handled just bit later with the Contains Clause processing */
         end
         else if left(mask,1) = fsSeparator
            then startingAt = filespec('drive',topLevelDir)    /* fixated on the Drive */
            else do
               /* signal a fixed search from our starting directory */
               mask = topLevelDir||strip(mask,'L',fsSeparator)
               startingAt = ''
            end
      end

      if dbg >= 2 then do
         if ContainsClause = 1
            then j = '"Contains" '
            else j = ''

         say 'currently in:' rootDir 'processing' j||'mask' maskParts.m
         if NotMask = 1
            then say '  '||AndMask||'filtering rules: NOT' startingAt||mask
            else say '  '||AndMask||'filtering rules:' startingAt||mask
         say 'Enter S to Stop, T to Trace, otherwise continue...'
         pull response
         if translate(left(response,1)) = 'S' then exit 1
         if translate(left(response,1)) = 'T' then trace("?R")
      end

      /* if this mask is specifying the subdirectory must (or must not) contain
         at least 1 of the files in the mask */
      if ContainsClause = 1 then do
         /* contains clause syntax and meaning, where '!' inverts the test:
               (mask)            current directory contains a file
               (path\mask)       initial starting directory + path contains a file
               (.\path\mask)     path relative to where we are contains a file
               (\path\mask)      hard path at top of the drive contains a file
         */
         startingAt = rootDir || fsSeparator
         if filespec('path',mask) <> '' then do
            /* we must consider how the 'path' affects the searching */
            if left(mask,1) = '.' then do
               /* it is relative to where we are presently in the directory tree,
                      so we remove the '.\' which means the current directory.
                      Note, a '..\' syntax is not supported */
               mask = substr(mask,pos(fsSeparator,mask)+1)
            end
            else if left(mask,1) = fsSeparator
               then startingAt = filespec('drive',topLevelDir)    /* fixated on the Drive */
               else startingAt = topLevelDir || fsSeparator       /* fixated where we started */
         end

         if OpSys = 'UNIX' & substr(tAttrib,3,1) = '-'
            /* Hidden files work differently on Unix/Linux */
            then rc = SysFileTree(startingAt || mask ,files,options,substr(tAttrib,1,2)||'*'||substr(tAttrib,4))
            else rc = SysFileTree(startingAt || mask ,files,options,tAttrib)
          fileCount = files.0
          drop files.
          if dbg >= 2 then say '      found' fileCount 'matching files.'

         /* if NO files were found */
         if fileCount = 0 then do
            /* Then IF the "not" condition is false, then we must find something to continue on */
            if NotMask = 0 then leave
         end
         else /* some files were found, so ... */
            if NotMask = 1 then leave     /* if we didn't want to find them, then we leave the loop */

         /* we've processed the 'contains' clause and can conntinue to the next mask */
         iterate
      end

      /* any path hard coded to the root is processed only once */
      if currentRecursionLevel > 1 & left(mask,1) = fsSeparator & ExcludeMask = 0
         then iterate

      /* can the system query handle the type of mask we were given? */
      if AndMask <> '' then do
         /* filter our current results ('allFiles.') against the mask
            returning the sucessful candidates (in 'files.'
            creating a new consolidated files list with the results */
         if consolidatedFiles.0 > 0 then do
            rc = SysStemCopy(consolidatedFiles,allFiles)
            drop consolidatedFiles.
            consolidatedFiles.0 = 0
            fileCount = filterAllFiles(mask)
          end
          else do
             fileCount = 0
             files.0 = fileCount
          end
      end
      else if filespec('path',maskParts.m) = '' & NotMask = 0 then do
         /* Hidden files work differently on Unix/Linux */
         if OpSys = 'UNIX' & substr(tAttrib,3,1) = '-'
            then rc = SysFileTree(startingAt || mask ,files,options,substr(tAttrib,1,2)||'*'||substr(tAttrib,4))
            else rc = SysFileTree(startingAt || mask ,files,options,tAttrib)
          fileCount = files.0
      end
      else do  /* else there is a path or a Not condition to process */
         if filespec('path',maskParts.m) <> '' & left(maskParts.m,1) <> '.' then do
            /* there are some parts of a specific path to match.
               If starts with a '\' then hard path on the drive,
                  else hard path in our initial starting directory */
            if left(maskParts.m,1) = fsSeparator
               then mask = filespec('drive',topLevelDIr)||maskParts.m
               else mask = filespec('drive',topLevelDIr)||filespec('path',topLevelDIr||fsSeparator||'.')||maskParts.m
         end

         /* if there is a path to consider */
         if filespec('path',mask) <> '' then do
            /* gather all the directories that match into the path */
            MatchingDirs.0 = 1
            MatchingDirs.1 = '. . . .' startingAt

            /* special case we may be able to optimize the query */
            if filespec('drive',mask) <> '' then do
               mask = substr(mask,3)
               startingAt = filespec('path',mask)

               /* first wild card position */
               i = pos('*',startingAt)
               j = pos('?',startingAt)
               if i = 0
                  then i = j

               /* update the starting point and mask to optimize the query */
               if i > 0 then do
                  j = lastpos(fsSeparator,left(startingAt,i))
                  startingAt = left(startingAt,j)
               end
               mask = substr(mask,length(startingAt)+1)
               MatchingDirs.1 = '. . . .' startingAt
            end

            rest = filespec('path',mask)
            do until rest = ''
               j = pos(fsSeparator,rest)
               if j > 0 then do
                  ques = left(rest,j)
                  rest = substr(rest,j+1)
               end
               else do
                  ques = rest
                  rest = ''
               end

               /* Enumerate the subdirectories that fit this part of the mask */
               drop tempList.
               drop files.
               tempList.0 = 0
               files.0 = 0
               do i = 1 to MatchingDirs.0
                  if files.0 > 0
                     then rc = SysStemCopy(files,tempList,1,tempList.0+1)
                  parse value MatchingDirs.i with . . . . prev
                  prev = strip(strip(prev),'T',fsSeparator)||fsSeparator
                  rc = SysFileTree(prev||strip(ques,'T',fsSeparator),files,'D',tAttrib)
               end

               /* finish consolidating the list */
               if files.0 > 0
                  then rc = SysStemCopy(files,tempList,1,tempList.0+1)

               /* put the new directories into the Matching Directories list */
               drop MatchingDirs.
               if tempList.0 > 0
                  then rc = SysStemCopy(tempList,MatchingDirs,1,1)
                  else MatchingDirs.0 = 0
            end
            drop tempList.
if gblTrace <> ''
   then say MatchingDirs.0 'Directory Matches'

            /* process each matching directory gathering all appropriate files */
            do i = 1 to MatchingDirs.0
if gblTrace <> '' then do
   say i MatchingDirs.1
   trace("?R")
end
               aSubDir = subword(MatchingDirs.i,5)||fsSeparator
               drop files.
               if NotMask = 0
                  then rc = SysFileTree( aSubDir||filespec('name',mask) ,files,options,tAttrib)
                  else rc = SysFileTree( aSubDir||'*' ,files,options,tAttrib)
               if files.0 > 0
                  then rc = SysStemCopy(files,allFiles,1,allFiles.0+1)
            end

            /* now filter allFiles as a "And" with the mask */
            saveAndMask = AndMask
            AndMask = 'And '
            fileCount = filterAllFiles(mask)
            AndMask = saveAndMask
            drop allFiles.
            allFiles.0 = 0
         end
         /* else when SysFileTree can not handle NOT Matching nor single match character rules */
         else if NotMask = 1 | pos('?',maskParts.m) > 0
            /* use our own filter logic */
            then fileCount = filterAllFiles(mask)
         else do
            /* Hidden files work differently on Unix/Linux */
            if OpSys = 'UNIX' & substr(tAttrib,3,1) = '-'
               then rc = SysFileTree( mask ,files,options,substr(tAttrib,1,2)||'*'||substr(tAttrib,4))
               else rc = SysFileTree( mask ,files,options,tAttrib)
            fileCount = files.0
         end
      end

      /* if we got some new files */
      if fileCount > 0 then do      /* merge new ones with the old ones */
         rc = SysStemCopy(files,consolidatedFiles,1,consolidatedFiles.0+1)
      end
   end

   drop files.
   files.0 = 0

   /* As necessary, sort the results so we can remove duplicates */
   if consolidatedFiles.0 > 0 then do
      rc = SysStemSort(consolidatedFiles, 'A', 'I', 1, consolidatedFiles.0, 42, 290)

      /* copy the 'consolidated' results into 'files'. removing any duplicates */
      j = 1
      files.0 = j
      files.j = consolidatedFiles.1
      do i = 2 to consolidatedFiles.0
         lnA = files.j
         lnB = consolidatedFiles.i
         if lnA <> lnB then do
            j = j + 1
            files.j = lnB
            files.0 = j
         end
      end
      drop consolidatedFiles.
   end

   /* see if we need to correct for operating system limitations with files > 2 gig */
   fourGig = 4 * 1024 * 1024 *1024
   do i = 1 to files.0
      parse value files.i with . . size attr fn
      if substr(attr,2,1) <> 'D' then do             /* don't query size on a directory */
         rc = stream(strip(fn),'c','query size')
         if size <> rc & rc <> '' then do
            /* the position of the size string in the data line */
            wrongSize = size
            j = pos(size,files.i)
         
            if rc < 0 then do
               size = fourGig + rc
               if rc = -1 then do
                  /* we can't compute the real flie size, so put an asteric by it meaning > 4 gig */
                  size = (size+1)||'*'
               end
            end
            else size = rc
         
            /* replace the file size with the newly computed value */
            if length(size) > length(wrongSize) & right(size,1) <> '*'
               then j = j - (length(size) - length(wrongSize))
         
            files.i = left(files.i,j-1)||size||substr(files.i,j+length(size))
         end
       end
   end

   /* if we got some new files */
   if dbg >= 2 & fileCount > 0 then do
      say 'Found some files in:' rootDir
      do rc = 1 to files.0
         say '   ' files.rc
      end
   end

   /* should the results of this directory be sorted? */
   if sortBy <> '' then do
      sortBeginCol = 42                     /* assume sort by Name Ascending */
      sortEndCol = 290
      ordering = 'A'
      if right(sortBy,1) = 'R'
         then ordering = 'D'
      if left(sortBy,1) = 'D' then do       /* sort by Date and Time */
         sortBeginCol = 1
         sortEndCol = 19
      end
      else if left(sortBy,1) = 'T' then do  /* Very Unusual - Sort only by TIME */
         sortBeginCol = 12
         sortEndCol = 19
      end
      else if left(sortBy,1) = 'S' then do  /* Sort by Size */
         sortBeginCol = 20
         sortEndCol = 31
      end

      if files.0 > 1 then
         rc = SysStemSort(files, ordering, 'I', 1, files.0, sortBeginCol, sortEndCol)
   end

   /* Filter on Date/Time/Size as necessary and Format the results for output */
   Do i = 1 to files.0 while stream(abortSignalFileName,'c','query exists') = ''
      l = files.i
      parse var l dt tm size attr fname
      fname = strip(fname,'L')      /* remove any leading blanks */
      shortName = filespec('name',fname)

      /* if on Unix/Linux and requesting non hidden files,
                 then file name must not start with a '.'             */
      if OpSys = 'UNIX' & substr(tAttrib,3,1) = '-' & left(shortname,1) = '.'
         then iterate

      spot = pos(fname,l)-1

      if show_longname > 0 then do
         rc = SysGetEA(fname,'.LONGNAME','longname')
         if (rc = 0 & length(longname) > 4) then do
            fname = filespec('drive',fname)||filespec('path',fname)||delstr(longname,1,4)
         end
         else shortName = ''
      end

      /* filter by Modification date (and time) */
      if begin_date > 0 then do
         if pos('/',dt) > 0 then do
            parse var dt mm '/' dd '/' yy
            if yy <= 99 & yy >= 70
               then yy = yy + 1900
               else yy = yy + 2000
            compare_date = yy||'-'||right(100+mm,2)||'-'||right(100+dd,2)
         end
         else do
            compare_date = dt
         end

         select
           when comparator = '' then do
                if compare_date < begin_date then iterate
             end
           when comparator = 'E' then do
                if compare_date \= begin_date then iterate
             end
           when comparator = 'G' then do
                if compare_date < begin_date then iterate
             end
           when comparator = 'L' then do
                if compare_date > begin_date then iterate
             end
           otherwise if compare_date < begin_date then iterate
         end

         /* if an ending date was also specified */
         if end_date > 0 then do
           /* if must be less than or equal to this ending date */
           if compare_date > end_date then iterate
/*
           select
             when d_comparator2 = '' then do
                  if compare_date > end_date then iterate
               end
             when d_comparator2 = 'E' then do
                  if compare_date \= end_date then iterate
               end
             when d_comparator2 = 'G' then do
                  if compare_date < end_date then iterate
               end
             when d_comparator2 = 'L' then do
                  if compare_date > end_date then iterate
               end
             otherwise if compare_date > end_date then iterate
           end
*/
         end

         pm = 0
         if right(tm,1) = 'p' then do
            pm = 12
            tm = left(tm,length(tm)-1)
         end
         parse var tm hr ':' min ':' sec
         if t_precision = 'HOURS'
            then min = 0                      /* minutes are not important for this comparison */
         if t_precision <> 'SECONDS'
            then sec = 0                      /* seconds are not important for this comparision */

         seconds = (((hr + pm) * 60) + min) * 60 + sec
/*
trace("?R")
say fname dt tm
*/

         select
           when t_comparator = '' then do
                /* interpret the time comparator using date rules */
                if compare_date = begin_date then do
                   if comparator = 'L' then do
                      if seconds >= begin_time then iterate
                   end
                   else do
                      if seconds < begin_time then iterate
                   end
                end
                /* else the time component is irrelevant as the date argument
                   has already filtered the file's timestamp */
             end
           when t_comparator = 'E' then do
                if seconds \= begin_time then iterate
             end
           when t_comparator = 'G' then do
                if seconds < begin_time then iterate
             end
           when t_comparator = 'L' then do
                if seconds > begin_time then iterate
             end
           otherwise if seconds < begin_time then iterate
         end

         /* was there a date and time range specified? */
         if end_time > 0 then do
            /* interpret the time range considering the date range.
               Out of Date range already handled as is beginning time,
               so we are only considering the ending date and ending time. */
            if t_comparator = '' then do
               if compare_date = end_date
                  then if seconds > end_time then iterate
            end
            else select
               /* Unusual cases here, we want the TIMES within each day to be within that day's range.*/
               when t_comparator2 = 'E' then do
                    if seconds \= end_time then iterate
                 end
               when t_comparator2 = 'G' then do
                    if seconds < end_time then iterate
                 end
               when t_comparator2 = 'L' then do
                    if seconds > end_time then iterate
                 end
               otherwise if seconds < end_time then iterate
            end
         end

      end

      /* filter by Size */
      if s_comparator \= '' then do
         select
            when s_comparator = 'E' then do
                 if size \= size_arg then iterate
              end
            when s_comparator = 'G' then do
                 if size < size_arg then iterate
              end
            when s_comparator = 'L' then do
                 if size > size_arg then iterate
              end
            otherwise do
                 say 'Error in logic.  s_comparator value "'||s_comparator||'" is unprogrammed!'
              end
         end
      end

      /* reformat the output to enclose the file name in quotes in case of blanks */
      if showFullPath = 1
         then output = left(l,spot)||'"'||fname||'"'
         else output = left(l,spot)||'"'||filespec('NAME',fname)||'"'

      if bIncludeEA > 0 then do
         /* request to include the EA size in the output */
         parse var output dt tm size attr '"' fname '"'
         fname = strip(fname)

         /* get the size of the EAs */
         eaSize=4                  /* assumed EA header length */
         rc = SysQueryEAList(fname,eaList)
         if eaList.0 > 0 then do ea=1 to eaList.0
            theEAname = eaList.ea
            rc = SysGetEA(fname,theEAname,"theEAvalue")
                     /* +1 where EA Name length is stored
                        +2 for the bytes <llhh> for length of EA value
                        + lengths of the name and values + a /0 terminator for each */
            eaSize = eaSize +1 +2 + (length(theEAname)+1) + (length(theEAvalue) + 1)
         end
         else eaSize = 0
    
         if bIncludeEA = 2 & eaSize = 0      /* want only files with EAs and there are none */
            then iterate

         spot = pos(attr,output)
         output = left(output,spot-1) || right('      '||eaSize,6) || substr(output,spot)

      end

      spot = pos('"',output)-1

      if show_longname > 1 then
         output = left(l,spot)|| left(shortName||"            ",12) '"'||fname||'"'
      if countOnly < 0 then do
         if showFullNameOnly >= 1 then do
            parse var output with '"' fullPathName '"'
            if execTemplate = ''
               then output = fullPathName
            else do
               output = execTemplate
               spot = pos('%',output)
               do while spot > 0
                  /* substituting the parts of the file path name */
                  if substr(output,spot+1,1) = 'D'              /* drive only */
                     then rc = filespec('drive',fullPathName)
                  else if substr(output,spot+1,1) = 'R' then do /* relative path only */
                         rc = filespec('path',fullPathName)
                         if relPath = translate(substr(rc,1,length(relPath))) then do
                            rc = substr(rc,length(relPath)+1)
                            if rc = ''
                               then rc = '.'||fsSeparator
                         end
                     end
                  else if substr(output,spot+1,1) = 'P'         /* path only */
                     then rc = filespec('path',fullPathName)
                  else if substr(output,spot+1,1) = 'p'         /* little path, no trailing \ */
                     then rc = strip(filespec('path',fullPathName),'T',fsSeparator)
                  else if substr(output,spot+1,1) = 'N'         /* name only */
                     then rc = filespec('name',fullPathName)
                  else if substr(output,spot+1,1) = 'n' then do /* name only without extension */
                     rc = lastpos('.',filespec('name',fullPathName))
                     if rc > 0
                        then rc = left(filespec('name',fullPathName),rc-1)
                        else rc = filespec('name',fullPathName)
                  end
                  else rc = fullPathName

                  if rc = fullPathName   /* was it only a % versus a %D or %P or %N */
                     then output = left(output,spot-1)||rc||substr(output,spot+1)
                     else output = left(output,spot-1)||rc||substr(output,spot+2)

                  spot = pos('%',output)
               end
            end
         end

         /* if the '/EXEC' modifier option was requested */
         if showFullNameOnly >= 3 then do
            /* we will Execute the command */
            if right(translate(word(execTemplate,1)),4) = translate(exExtension)  /* typically '.CMD' or perhaps '.REX' */
               then output = '@call' output

            output = quoteString(output)       /* quote the string for the "interpret" command */
            interpret output
            thisRC = rc

            /* should we pause after each invocation so results can be checked? */
            if pauseOpt then do
               pauseNow = 1                                  /* assume we should pause */
               if execPauseRC <> "" then do                  /* pause on specific RC condition ? */
                  if left(execPauseRC,1) = '!' then do
                     if substr(execPauseRC,2) = thisRC
                        then pauseNow = 0
                  end
                  else do
                     if execPauseRC <> thisRC
                        then pauseNow = 0
                  end
               end
               if pauseNow = 1 then do
                 call charout ,'('||filespec('name',fullPathName)||') Press ENTER to continue...(or anything else to stop)'
                 pull rc
                 if rc <> '' then do
                    say '  ... Early exit request acknowledged.'
                    exit 2
                 end
               end
            end
         end
         else do     /* just output the formatted results */

            if pauseOpt then do
               /* considering a possible line wrap, count another line of output */
               linesNeeded = trunc(length(output)/cols)+1
               linesOutput = linesOutput + linesNeeded
               if linesOutput >= rows then do
                  /* place the prompt at the bottom and no extra linefeed */
                  call charout ,'Press ENTER to continue...(or anything else to stop)'
                  pull rc
                  if rc <> '' then do
                     say '  ... Early exit request acknowledged.'
                     exit 2
                  end
                  linesOutput = linesNeeded
               end
            end
            say strip(output)
         end
      end

      thisCount = thisCount + 1
      thisSize = thisSize + strip(size,'T','*')    /* ignore the asteric flagging sizes > 4 gig */

   End
End

if stream(abortSignalFileName,'c','query exists') <> '' then do
   say abortSignalFileName 'exists, so further operation is aborted!'
   rc = SysFileDelete(abortSignalFileName)
   exit 2
end

/* if to display the nested directory information */;
if nesting \= 0 & dirs.0 = 0 then do

   rc = SysFileTree(rootDir || fsSeparator||'*',subDirs,'O',"*+***")
   Do d = 1 to subDirs.0

      /* if there is a path to exclude and we are there */
      excludeSubdir = 0
      Do m = 1 to maskParts.0
         /* When mask is not a "contains" mask */
         mask = maskParts.m
         if left(mask,1) <> '(' then do
            /* Handle the '&' and '!' prefixes */
            AndMask = ''
            if left(mask,1) = '&' then do
               mask = substr(mask,2)
               AndMask = 'And '
            end

            NotMask = 0
            if left(mask,1) = '!' then do
               NotMask = 1
               mask = substr(mask,2)
            end

            if left(mask,2) = '.'||fsSeparator
               then mask = substr(mask,3)
            if filespec('path',mask) <> '' & AndMask <> '' & NotMask = 1 then do
               if strip(rootDir || fsSeparator || filespec('path',mask),'T',fsSeparator) = subDirs.d then do
                  excludeSubdir = 1
                  leave
               end
            end
         End
      End

      /* if this subdirectory is excluded from the recursion, skip it */
      if excludeSubdir = 1
         then iterate

      call listDirectory subDirs.d
      parse pull subCount ',' subSize
      if countOnly >= 0 & subCount >= countOnly then do
         say '"'||subdirs.d||'"' subCount 'entries' prettyNum(subSize) 'bytes'
      end
      if rollup > 0 then do
         thisCount = thisCount + subCount
         thisSize = thisSize + subSize
      end
   End
End

push thisCount||','||thisSize
currentRecursionLevel = currentRecursionLevel - 1
return

/*************************************
 proceedure to handle "NOT" conditions
           and '&' as "AND" conditions
 Requires NotMask and AndMask to be set
    for AndMask conditions,
       allFiles contains candidates
*************************************/

filterAllFiles:
   arg currentMask        /* assumption being currentMask has been translated to upper case for optimization */

   thisPassExcludeThese = (AndMask <> '' & notMask)
   if ExcludeMask | thisPassExcludeThese then do
      /* following was old code but we think as of 4/26/2025 maskP should just be startingAt 
      maskP = rootDir || fsSeparator
      if filespec('drive',currentMask) <> '' | left(currentMask,1) = fsSeparator
         then maskP = ''
      */
      maskP = startingAt

      if OpSys = 'UNIX' & substr(tAttrib,3,1) = '-'
         then rc = SysFileTree(maskP || currentMask ,ExcludeFiles,options,substr(tAttrib,1,2)||'*'||substr(tAttrib,4))
         else rc = SysFileTree(maskP || currentMask ,ExcludeFiles,options,tAttrib)
   end

   /* allFiles. will contain an existing candidate list when processing an 'And' condition. */
   if AndMask = '' then do
      /* Looking for NOT Matching, so list all files and then remove those that match */
      if OpSys = 'UNIX' & substr(tAttrib,3,1) = '-'
         then rc = SysFileTree(rootDir || fsSeparator || '*' ,allFiles,options,substr(tAttrib,1,2)||'*'||substr(tAttrib,4))
         else rc = SysFileTree(rootDir || fsSeparator || '*' ,allFiles,options,tAttrib)
   end

   files.0 = 0
   Do i = 1 to allFiles.0
      l = allFiles.i
      parse var l dt tm size attr fnamePath
      fnamePath = translate(strip(fnamePath,'L'))              /* fully qualified name is case independent */
      fname = fnamePath

traceIt = 0
if pos(translate(gblTrace),filespec('name',fname)) > 0 then do
   trace("?R")
   say 'Ready to trace file' fname
   traceIt = 1
end

      if filespec('path',currentMask) = ''
         then fname = filespec('name',fname)                   /* disregard drive & path parts */
      else if filespec('drive',currentMask) = ''               /* mask had No drive spec */
         then fname = substr(fname,length(startingAt)+1)

      /* are we excluding files that were found in the other specified directory? */
      if ExcludeMask | thisPassExcludeThese then do
         /* if the name matches irregardless of path information we will not include the file */
         foundIt = 0
         if ExcludeMask
            then fname = filespec('name',fname)

         do iEx = 1 to ExcludeFiles.0
            fEx = ExcludeFiles.iEx
            parse upper var fEx . . . . exFullP
            if ExcludeMask | filespec('path',currentMask) = ''
               then exFname = filespec('name',exFullP)
               else exFname = strip(exFullP)
            if fname = exFname & (thisPassExcludeThese | fnamePath <> exFullP) then do
               foundIt = 1
               leave
            end
         end

         if foundIt = 0 & NotMask then do
            /* allFiles.i was not found in ExcludeFiles.* and it was a NOT condition */
            rc = files.0 + 1
            files.rc = l
            files.0 = rc
         end
         else if foundIt & NotMask = 0 then do
            /* allFiles.i was found in ExcludeFiles.* and we wanted to include it */
            rc = files.0 + 1
            files.rc = l
            files.0 = rc
         end

         /* end of this special logic */
         iterate
      end

      /* process the mask character by character */
      maskP = 1
      fnameP = 1
      do while fnameP <= length(fname) & maskP <= length(currentMask)
         maskC = substr(currentMask,maskP,1)
         Select
            when maskC = '?' then do
               /* wild card match one character position only */
               fnameP = fnameP + 1
               maskP = maskP + 1
            end
            when maskC = '*' then do
if traceIt = 1 then trace("?R")
               /* find the position in the mask of the next wild card (if any) */
               consumeUpTo = pos('*',currentMask,maskP + 1)
               nextWildCardQuesP = pos('?',currentMask,maskP + 1)
               if consumeUpTo = 0 | (nextWildCardQuesP > 0 & nextWildCardQuesP < consumeUpTo)
                  then consumeUpTo = nextWildCardQuesP

               /* set the search string we are to match */
               if consumeUpTo > 0
                  then toMatch = substr(currentMask,maskP+1,consumeUpTo - maskP - 1)
                  else toMatch = substr(currentMask,maskP+1)

               /* if there is more to match */
               if toMatch <> '' then do
                  /* be greedy in matching a '*' wild card to the most we can grab */
                  rc = fnameP   /* remember how far we've gone thru the candidate */
                  fnameP = lastpos(toMatch,fname)
                  if fnameP > 0 & fnameP > rc
                     then fnameP = fnameP + length(toMatch)
                     else leave
               end
               else fnameP = length(fname)+1     /* final wild card matches all that's left */

               maskP = maskP + 1 + length(toMatch)
            end
            otherwise do
               /* if not a matching character, we are done and failed to match */
               if substr(fname,fnameP,1) <> maskC
                  then leave

               fnameP = fnameP + 1
               maskP = maskP + 1
            end
         End
      end

if traceIt = 1 then trace("?R")
      /* if it DID NOT match, i.e.:
         More "fname" left, or more "mask", or not at an ending '*' in the mask,
         this one should be included  as a "no match" */
      if length(fname) >= fnameP | length(currentMask) > maskP | (substr(currentMask,maskP,1) <> '*' & maskP = length(currentMask)) then do
         if NotMask = 1 then do
            rc = files.0 + 1
            files.rc = l
            files.0 = rc
         end
      end
      else if NotMask = 0 then do
         rc = files.0 + 1
         files.rc = l
         files.0 = rc
      end
   end

   /* signal nothing of importance in 'allFiles' */
   allFiles.0 = 0

return files.0

/* this routine explains what the specified masks will attempt to do */
ExplainMasks:
   say ''
   say 'Processing order of mask: "'||maskArg||'"'
   say '    Analyzing first part of the mask, operation begins at:' topLevelDir
   say ''
   orderWords = 'First, Secondly Next Then'
   do m = 1 to maskParts.0
      if m < words(orderWords)-1
         then Phrase = word(orderWords,m)
         else Phrase = word(orderWords,words(orderWords))

      mask = maskParts.m
      maskInternals = mask
      NotPhrase = ''
      AndPhrase = ''
      Clause = 'match'
      kind = 'files'

      if left(maskInternals,1) = '(' then do
         Clause = 'CONTAIN 1 or more'
         kind = 'a directory'
         parse var maskInternals . '(' maskInternals ')' .
      end

      if left(maskInternals,1) = '&' then do
         AndPhrase = ' AND'
         maskInternals = substr(maskInternals,2)
      end

      if left(maskInternals,1) = '!' then do
         NotPhrase = ' NOT'
         maskInternals = substr(maskInternals,2)
      end

      if left(maskInternals,1) = '=' then do
         Clause = ' HAVE'
         maskInternals = substr(maskInternals,2)
      end

      say m||'.' Phrase mask 'means'||AndPhrase kind 'must'||NotPhrase Clause maskInternals
      msg2 = ''
      msg3 = ''
      posSeparator = pos(fsSeparator,maskInternals)
      if posSeparator > 0 then do
         if posSeparator = 1
            then posSeparator = pos(fsSeparator,maskInternals,posSeparator+1)
         firstSubdir = substr(maskInternals,1,posSeparator-1)
         msg1 = 'As it starts with "'.'" it is relative to nested subdirectories in the tree'
         ch = left(firstSubdir,1)
         if ch = fsSeparator
            then msg1 = 'Since it starts with "'||fsSeparator||'", it is processed ALWAYS from the root' filespec('drive',rootDrive)||fsSeparator
         else if ch <> '.' then do
            msg1 = 'As it has a path "'||firstSubdir||'" that is not relative ".\",'
            msg2 = 'It is searched under initial directory' topLevelDir
            msg3 = "AND it's tree is not recursively traversed!"
         end
      end
      else do
          msg1 ='As it contains no path information,'
          msg2 ='each subdirectory is searched as the tree is traversed.'
      end

      say '    ' msg1
      if msg2 <> '' then say '         ' msg2
      if msg3 <> '' then say '         ' msg3
   end

   say 'Enter "Q" to Quit, Otherwise proceed...'
   pull response
   if translate(left(response,1)) = 'Q'
      then exit 2
return
