/*
 * Filename: HTML2HLP.cmd
 *   Author: JAN-ERIK
 *  Created: Sat Mar 22 2014
 *  Purpose: Tool to convert A HTML page to IPF
 *  Changes: Sat Oct 13 2018 - Flattened recursion levels, fixed footnote, fixed load library unzip32, unordered list by default/simple list with none set as list-style-type, only compact when defined, .inf-file
 */

/* Load Library */
IF RxFuncQuery( 'UZLoadFuncs' ) THEN
DO
    CALL RxFuncAdd 'UZLoadFuncs', 'unzip32', 'UZLoadFuncs'
    CALL UZLoadFuncs
END

/* Load RexxUtil Library */
IF RxFuncQuery('SysLoadFuncs') THEN
DO
    CALL RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
    CALL SysLoadFuncs
END
CALL rxOut 0, ' '
CALL SysCls

PARSE SOURCE . . this
PARSE VALUE REVERSE( this ) WITH .'.'this'\'.
this = REVERSE( this )

/* Pass 1: Process input parameter and read the html-page */
IF LENGTH( STRIP( ARG(1) ) ) = 0 THEN RETURN Usage()
CALL SysSleep 1
CALL rxOut 0, 'Pass: 1 ( of 9 )'
CALL rxProgress 2, 0, 1 / 9, 1
PARSE VALUE REVERSE( ARG(1) ) WITH .'.'outfile.5
outfile.5 = REVERSE( outfile.5 )
outfile.1 = outfile.5||'_split.html'
outfile.2 = outfile.5||'_style.html'
outfile.3 = outfile.5||'.ipf'
outfile.4 = outfile.5||'.hlp'
outfile.5 = outfile.5||'.inf'

IF STREAM( ARG(1), 'C', 'QUERY EXISTS' ) = '' THEN RETURN 2
f_size = STREAM( ARG(1), 'C', 'QUERY SIZE' )
IF f_size = 0 THEN RETURN 3
indata = CHARIN( ARG(1), 1, f_size )
CALL STREAM ARG(1), 'C', 'CLOSE'
CALL rxProgress 2, 1 / 9, 1 / 9, 1

/* Pass 2: Process the support libraries */
CALL SysSleep 1
libs = 'char tag style color'
newln = D2C(13)||D2C(10)
DO i = 1 TO WORDS( libs )
    CALL rxProgress 2, 1 / 9 + i / ( WORDS( libs ) / 9 ), 2 / 9, 1
    rc = UZUnZipToVar( 'html2hlp.add', SUBWORD( libs, i, 1 )||'s', SUBWORD( libs, i, 1 )||'s.' )
END

/* Pass 3: Process the html file and split it up into an array */
CALL SysSleep 1
curr_tag = parse_next_tag( indata, LENGTH( indata ),,, 3 / 9, 1 / 9 )

/* Pass 4: Write it to file */
CALL SysSleep 1
f_array = 'input'
expose_list = f_array||'. newln'
rc = rxWrite2File( outfile.1, 4 / 9, 1 / 9 )

/* Pass 5: Prepare output */
CALL SysSleep 1
output.1 = '.* Generated by '||this||' on '||DATE()||'.'
CALL chk_tags_2_parse 5 / 9, 1 / 9

/* Pass 6: Process styles */
CALL SysSleep 1
CALL applystyle "css", 6 / 9, 1 / 9

/* Pass 7: Write it to file */
CALL SysSleep 1
rc = rxWrite2File( outfile.2, 7 / 9, 1 / 9 )

/* Pass 8: Write the output as an ipf file */
CALL SysSleep 1
f_array = 'output'
expose_list = f_array||'. newln'
rc = rxWrite2File( outfile.3, 8 / 9, 1 / 9 )

CALL UZDropFuncs

/* Pass 9:  Use ipfc to create the hlp file */
CALL SysSleep 1
CALL rxOut 0, 'Pass: 9 ( of 9 )'
CALL rxOut 1, ' '
SAY COPIES( newln, 2 )
'@IPFC '||outfile.3||' '||outfile.4
IF rc <> 0 THEN
   '@IPFC '||outfile.3||' '||outfile.4||'|more'
ELSE
   '@IPFC '||outfile.3||' '||outfile.5
CALL rxProgress 2, 1
PARSE PULL
RETURN rc

rxWrite2File: PROCEDURE EXPOSE (expose_list) (f_array)
    CALL SysFileDelete ARG(1)
    DO i = 1 TO VALUE( f_array'.0' )
        j = i - 1
        CALL rxOut 0, 'Pass: '||FORMAT( ARG(2)/ARG(3),, 0 )||' ( of '||FORMAT( 1/ARG(3),, 0 )||' )'
        CALL rxWorking 1, SUBWORD( VALUE( f_array'.i' ), 1, 1 )
        CALL rxProgress 2, ARG(2) + i / VALUE( f_array'.0' ) * ARG(3), ARG(2) + ARG(3), 1
        IF VALUE( f_array'.i' ) <> newln THEN
            IF f_array = 'output' THEN
            DO
                IF VALUE( f_array'.j' ) = newln||'.br' & POS( newln, VALUE( f_array'.i' ) ) <> 1 THEN
                    CALL CHAROUT ARG(1), newln
                IF output.i <> newln THEN
                    CALL CHAROUT ARG(1), VALUE( f_array'.i' )
            END
            ELSE IF VALUE( f_array'.i.lvl' ) > 0 THEN
                CALL LINEOUT ARG(1), COPIES( ' ', VALUE( f_array'.i.lvl' ) )||STRIP( VALUE( f_array'.i' ) )
            ELSE
                CALL LINEOUT ARG(1), STRIP( VALUE( f_array'.i' ) )
    END
    CALL STREAM ARG(1), 'C', 'CLOSE'
RETURN 0

metadata: PROCEDURE EXPOSE __meter. input. output. chars. i
    PARSE VALUE input.i WITH '<meta'.'"'meta_descr'" content="'meta_value'"'.
    IF LENGTH( meta_descr ) = 0 & LENGTH( meta_value ) = 0 THEN
        PARSE VALUE input.i WITH '<meta content="'meta_value'"'.'"'meta_descr'"'.
    IF LENGTH( meta_descr ) > 0 & LENGTH( meta_value ) > 0 THEN
        output.i = output.i||' '||adjust_chars( meta_descr||': '||meta_value )
RETURN 0

calc_table_length: PROCEDURE EXPOSE __meter. input. output. chars. i
    IF TRANSLATE( input.i ) = '</TABLE>' THEN RETURN 0
    maxi.0 = 0
    k = 0
    DO j = i + 1 TO input.0
        IF TRANSLATE( input.j ) = '<BR>' THEN input.j = ''
        IF TRANSLATE( input.j ) = '</TABLE>' THEN LEAVE j
        IF TRANSLATE( input.j ) = '<TR>' THEN DO
            k = 0
            ITERATE j
        END
        IF POS( '<TD', TRANSLATE( input.j ) ) > 0 THEN
        DO
            k = k + 1
            ITERATE j
        END
        IF LEFT( input.j, 1 ) = '<' & RIGHT( input.j, 1 ) = '>' THEN
            ITERATE j
        IF \DATATYPE( maxi.k, 'W' ) THEN
            maxi.k = 0
        maxi.k = MAX( maxi.k, LENGTH( adjust_chars( input.j,, 'ASCII' ) ) )
        maxi.0 = MAX( k, maxi.0 )
    END
    retval = " cols='"
    IF POS( 'BORDER="0"', TRANSLATE( input.i ) ) > 0 THEN retval = ' frame=none rules=none'||retval
    ELSE IF POS( 'BORDER=0', TRANSLATE( input.i ) ) > 0 THEN retval = ' frame=none rules=none'||retval
    DO k = 1 TO maxi.0
        retval = retval||maxi.k||' '
    END
    output.i = output.i||STRIP( retval, 'T' )||"'."
RETURN 0

GetLinkType: PROCEDURE EXPOSE link.
    PARSE ARG link.ref
    IF POS( ':', link.ref ) > 0 THEN
        PARSE VALUE ARG(1) WITH link.scheme':'link.path'#'.
    ELSE DO
        link.scheme = ''
        link.path = ''
    END
    PARSE VALUE ARG(1) WITH link.url'#'link.target
    ext = '.CMD .EXE .INF .HLP'
    link.scheme.clean = SPACE( TRANSLATE( link.scheme,, TRANSLATE( XRANGE(),, XRANGE( 'a', 'z' )||XRANGE( 'A', 'Z' ) ) ), 0 )
    retval = 0 /* Unknown */
    IF LENGTH( link.scheme ) = 1 THEN
        IF link.scheme.clean = link.scheme THEN
            retval = FileExist( link.url ) /* Local file */
        ELSE retval = -1 /* Unknown scheme*/
    ELSE IF POS( '\\', link.scheme ) = 1 THEN
        retval = 4 + FileExist( link.url ) /* Network */
    ELSE DO
        IF LENGTH( link.target ) > 0 THEN
            retval = 9 /* Internal target */
        ELSE IF link.scheme.clean = link.scheme & LENGTH( link.scheme ) > 0 THEN
            retval = 8 /* External link */
        ELSE IF FileExist( link.url ) = 1 THEN
            retval = 10 /* Path */
        ELSE IF DATATYPE( link.url, 'S' ) THEN
            retval = 11 /* Target */
    END
    DO i = 1 TO WORDS( ext )
        IF POS( SUBWORD( ext, i, 1 ), TRANSLATE( link.url ) ) > 0 THEN
        DO
            retval = ( LENGTH( link.target ) > 0 ) * WORDS( ext ) + 11 + i /*executable, script, inf-file, helpfile*/
            LEAVE i
        END
    END
RETURN retval

EncodeTarget: PROCEDURE
    retval = 0
    input = LEFT( ARG(1), MIN( LENGTH( ARG(1) ), 250 ) )
    DO i = 1 TO LENGTH( input ) BY 2
        retval = retval + C2D( SUBSTR( input, i, 2 ) )
    END
RETURN retval // 65535

chk_hd: PROCEDURE EXPOSE __meter. input. output. i newln
    PARSE VALUE input.i WITH '<'tag' '.
    PARSE VALUE tag WITH tag'>'.
    IF POS( '</'||STRIP( tag, 'L', '/' )||'>', input.i ) THEN RETURN 0 /* End Tag */
    DO j = i + 1 TO input.0
        IF POS( '<', input.j ) = 0 THEN
        DO
            output.i = output.i||' id='||EncodeTarget( SPACE( TRANSLATE( input.j,, TRANSLATE( XRANGE(),, XRANGE( 'a', 'z' )||XRANGE( 'A', 'Z' )||XRANGE( '0', '9' ) ) ), 0 ) )||'.'
            RETURN 0
        END
    END
RETURN 1

chk_lnk: PROCEDURE EXPOSE __meter. input. output. i newln
    IF input.i = '</a>' THEN RETURN 0
    a_name = parse_xml_tags( input.i, 'a', ' name', 5 ) /*blabla*/
    a_href = parse_xml_tags( input.i, 'a', ' href', 5 ) /*file_now OR run.cmd or http://www...*/
    IF LENGTH( a_href ) > 0 THEN
        linktype = GetLinkType( a_href )
    ELSE IF LENGTH( a_name ) > 0 THEN
    DO
        linktype = 0
        link.target = a_name
    END
    level = input.i.lvl
    link.target.encoded = EncodeTarget( SPACE( TRANSLATE( link.target,, TRANSLATE( XRANGE(),, XRANGE( 'a', 'z' )||XRANGE( 'A', 'Z' )||XRANGE( '0', '9' ) ) ), 0 ) )
    IF 9 <= linktype & linktype < 11 | linktype = 14 THEN
    DO /* hd */
        output.i = ":link reftype=hd refid='"||link.target.encoded||"'"
        IF 13 < linktype THEN
            output.i = output.i||" database='"||adjust_chars( link.url )||"'"
        output.i = output.i||'.'
        j = i + 1
        output.j = input.j
        IF linktype = 12 THEN
            output.j = newln||":i1 id="||link.target.encoded||"."
    END
    ELSE IF linktype = 11 THEN
    DO /* fn */
       output.i = ":link reftype=fn refid="||link.url||"."
    END
    ELSE IF linktype > 17 THEN
    DO /* inform */
        output.i = ':link reftype=inform res='||link.target.encoded||'.'
        level = -1
    END
    ELSE IF linktype <> 0 THEN DO
        /* launch */
        IF linktype < 8 THEN
            output.i = ":link reftype=launch object='cmd.exe' data='/c open "||link.url||"'"
        ELSE IF linktype = 8 THEN
            output.i = ":link reftype=launch object='netscdde.exe' data='-Xn "||link.ref||"'"
        ELSE DO
            output.i = ":link reftype=launch object='"||LEFT( link.url, MIN( LENGTH( link.url ), 47 ) )||"'"
            IF LENGTH( link.target ) > 0 THEN
                output.i = output.i||" data='"||link.target||"'"
        END
        output.i = output.i||'.'
    END
    IF level <> -1 THEN
    DO j = i + 1 TO input.0
        IF POS( '</a>', input.j ) = 1 & input.j.lvl = level THEN DO
            IF LENGTH( a_name ) = 0 THEN
                output.j = output.j||':elink.'
            LEAVE j
        END
    END
    IF linktype = 11 THEN
    DO
        tag = 'a'
        DO j = j TO input.0
            IF LEFT( input.j, 1 ) = '<' THEN DO
                tag_id = parse_xml_tags( input.j, 'a', ' name', 5 )
                IF LENGTH( tag_id ) > 0 & tag_id = link.url THEN
                DO
                   output.j = ':fn id='tag_id'.:p.'
                    level = input.j.lvl
                    DO k = j + 1 TO input.0
                        IF POS( '</'||tag||'>', input.k ) = 1 & input.k.lvl = level THEN DO
                            output.k = ':efn.'
                            LEAVE j
                        END
                    END
                    CALL LINEOUT 'STDERR', 'Warning! Footnote without "end", start: '||j
                END
            END
        END

    END
    ELSE IF LENGTH( a_name ) > 0 & LEFT( output.i, 3 ) <> ':fn' THEN
    DO
        IF output.i <> newln THEN
            output.i = output.i||newln
        output.i = output.i||":i1 id="||link.target.encoded||"."||link.target
    END
RETURN 0

applystyle: PROCEDURE EXPOSE __meter. input. output. i from. tgt. colors. newln
    p_align = ''
    DO i = 1 TO input.0
        IF LEFT( input.i, 2 ) = '</' | POS( '<', input.i ) = 0 THEN ITERATE i
        CALL rxOut 0, 'Pass: '||FORMAT( ARG(2)/ARG(3),, 0 )||' ( of '||FORMAT( 1/ARG(3),, 0 )||' )'
        CALL rxWorking 1, input.i
        CALL rxProgress 2, ARG(2) + i / input.0 * ARG(3), ARG(2) + ARG(3), 1
        PARSE VALUE input.i WITH '<'tag' '.
        PARSE VALUE tag WITH tag'>'.
        IF TRANSLATE( tag ) = 'BR' THEN ITERATE i
        level = input.i.lvl
        j = i + 1
        text_align = getstyle( input.i, 'text-align' )
        IF LENGTH( STRIP( text_align ) ) > 0 & tag <> 'img' THEN
            output.i = CHANGESTR( ':p.', output.i, ':lines align='||text_align||'.' )
        color = getstyle( input.i, 'color' )
        italic = getstyle( input.i, 'font-style' )
        italic = ( italic <> 'normal' & italic <> '' )
        underline = ( getstyle( input.i, 'text-decoration' ) = 'underline' )
        bold = getstyle( input.i, 'font-weight' )
        IF POS( 'bold', bold ) THEN bold = 1
        ELSE IF DATATYPE( bold, 'W' ) THEN
            IF bold > 600 THEN bold = 1
            ELSE bold = 0
        ELSE bold = 0
        IF 'UL' = TRANSLATE( tag ) THEN
        DO
            IF POS( 'COMPACT', TRANSLATE( input.i ) ) > 0 THEN
               output.i = CHANGESTR( ':ul.', output.i, ':ul compact.' )
            IF TRANSLATE( getstyle( input.i, 'list-style-type' ) ) = 'NONE' THEN
            DO
                output.i = CHANGESTR( ':ul', output.i, ':sl' )
                DO j = i + 1 TO input.0
                   IF POS( '</'tag'>', input.j ) > 0 THEN
                   DO
                      output.j = CHANGESTR( ':eul.', output.j, ':esl.' )
                      j = input.0
                   END
                END
            END
        END
        IF 'IMG' = TRANSLATE( tag ) THEN
        DO
            img_src = parse_xml_tags( input.i, 'img', ' src', 5 )
            retval = GetLinkType( img_src )
            IF link.path = '' THEN
                retval = GetLinkType( DIRECTORY()||'\'||img_src )
            IF retval = 1 THEN
            DO
                PARSE VALUE REVERSE( link.url ) WITH .'.'img_src
                retval = GetLinkType( REVERSE( img_src )||'.bmp' )
            END
            artwork_name = " name='"||link.url||"'"
            artwork_align = text_align
            IF LENGTH( artwork_align ) > 0 & artwork_align <> 'justified' THEN
                artwork_align = " align="||artwork_align
            ELSE artwork_align = ""
            artwork_runin = parse_xml_tags( input.i, 'img', ' align', 5 )
            IF LENGTH( artwork_runin ) = 0 THEN
                artwork_runin = " runin"
            ELSE artwork_runin = ''
            artwork_width = getstyle( input.i, 'width' )
            artwork_height = getstyle( input.i, 'height' )
            artwork_fit = ''
            IF LENGTH( artwork_width ) > 0 & LENGTH( artwork_height ) > 0 THEN
                IF artwork_width = '100%' & artwork_height = '100%' THEN
                    artwork_fit = ' fit'
            output.i = output.i||artwork_name||artwork_align||artwork_runin||artwork_fit||'.'
            ITERATE i
        END
        ELSE IF LENGTH( color ) > 0 THEN
        DO
            IF POS( 'rgb(', color ) > 0 THEN
            DO
                PARSE UPPER VALUE color WITH 'RGB('red','green','blue')'
                hex_val = D2X( STRIP( red ), 2 )||D2X( STRIP( green ), 2 )||D2X( STRIP( blue ), 2 )
            END
            ELSE IF POS( '#', color ) > 0 THEN
                PARSE VALUE color WITH '#'hex_val
            w_pos = WORDPOS( hex_val, colors.1 )
            IF w_pos > 0 THEN
                output.i = output.i||":color fc="||SUBWORD( colors.2, w_pos, 1 )||"."
        END
        IF bold & underline THEN
            output.i = output.i||':hp7.'
        ELSE IF italic & underline THEN
            output.i = output.i||':hp6.'
        ELSE IF underline THEN
            output.i = output.i||':hp5.'
        ELSE IF bold & italic THEN
            output.i = output.i||':hp3.'
        ELSE IF bold THEN
            output.i = output.i||':hp2.'
        ELSE IF italic THEN
            output.i = output.i||':hp1.'

        DO j = i + 1 TO input.0
            IF POS( '</', input.j ) > 0 THEN
                PARSE VALUE input.j WITH '</'tag'>'.
            ELSE ITERATE j
            IF input.j.lvl = level THEN
            DO
                IF bold & underline THEN
                    output.j = output.j||':ehp7.'
                ELSE IF italic & underline THEN
                    output.j = output.j||':ehp6.'
                ELSE IF underline THEN
                    output.j = output.j||':ehp5.'
                ELSE IF bold & italic THEN
                    output.j = output.j||':ehp3.'
                ELSE IF bold THEN
                    output.j = output.j||':ehp2.'
                ELSE IF italic THEN
                    output.j = output.j||':ehp1.'
                italic = 0;underline = 0;bold = 0;
                IF text_align <> '' THEN
                    output.j = ':elines.'
                ITERATE i
            END
        END
    END
RETURN 0

textstyle: PROCEDURE EXPOSE __meter. input. output. i from. tgt. colors. newln
    PARSE ARG set_style, styles
    fnt.1 = 'Courier'
    fnt.2 = 'Helv'
    fnt.3 = 'Tms Rmn'
    fnt.4 = 'System Monospaced'
    fnt.5 = 'System Pro'
    fnt.0 = 5
    level = input.i.lvl
    PARSE VALUE input.i WITH '<'tag' '.
    PARSE VALUE tag WITH tag'>'.
    DO j = i TO input.0
        IF set_style = 'clear' THEN
        DO
            IF POS( '<'||tag, input.j ) = 1 & input.j.lvl = level THEN
            DO
                output.j = ''
                LEAVE j
            END
        END
        IF set_style = 'color' | set_style = 'font' THEN
        DO
            IF POS( 'color="', input.j ) > 0 THEN
                input.j = injectstyle( input.j, 'color:',, 'color' )
        END
        IF tag = 'tt' THEN
        DO
            IF i = j THEN
                input.j = '<font style="font-family: System Monospaced;">'
            ELSE IF POS( '</'||tag, input.j ) = 1 & input.j.lvl = level THEN DO
                input.j = '</font>'
                tag = 'font'
                j = i
            END
        END
        IF tag = 'font' THEN
        DO
            IF POS( 'face="', input.j ) > 0 THEN
                input.j = injectstyle( input.j, 'font-family:',, 'face' )
            IF POS( 'size="', input.j ) > 0 THEN DO
                PARSE VALUE input.j WITH .'size="'fsize'"'
                wsize = '-2 7 -1 8 0 10 +1 12 +2 15 +3 20 +4 25'
                wpos = WORDPOS( fsize, wsize )
                IF wpos = 0 THEN wpos = 3
                input.j = injectstyle( input.j, 'font-size:', SUBWORD( wsize, wpos + 1, 1 ), 'size' )
            END
            ELSE IF POS( '</font>', input.j ) = 1 & input.j.lvl = level THEN DO
                output.j = ''
                LEAVE j
            END
        END
        IF set_style = 'center' | set_style = 'img' | set_style = 'align' THEN
        DO
            IF tag = 'center' THEN stl = 'center'
            ELSE IF i = j THEN
                PARSE VALUE input.j WITH '<'(tag).'style='.'text-align:'stl';'.
            IF POS( '<P', TRANSLATE( input.j ) ) = 1 & LENGTH( stl ) > 0 THEN
                input.j = injectstyle( input.j, 'text-align:', stl )
            ELSE IF POS( '<IMG', TRANSLATE( input.j ) ) = 1 THEN
            DO
                input.j = injectstyle( input.j, 'width:',, 'width' )
                input.j = injectstyle( input.j, 'height:',, 'height' )
                IF LENGTH( stl ) > 0 THEN
                    input.j = injectstyle( input.j, 'text-align:', stl )
                IF set_style = 'img' THEN stl = ''
            END
            ELSE IF POS( '</center>', input.j ) = 1 & input.j.lvl = level THEN DO
                stl = ''
                output.j = ''
                LEAVE j
            END
        END
    END
RETURN 0

getstyle: PROCEDURE
    PARSE ARG text, style_type
    xtra = ''
    PARSE VALUE text WITH '<'tag' '.
    PARSE VALUE tag WITH tag'>'.
    PARSE VALUE text WITH '<'(tag).'style="'stl'"'.
    PARSE VALUE stl WITH (style_type)':'mod_val';'
RETURN STRIP( mod_val )

injectstyle: PROCEDURE EXPOSE __meter. link newln chars.
    PARSE ARG text, style_type, style_setting, mod_type
    xtra = ''
    PARSE VALUE text WITH '<'tag' '.
    PARSE VALUE tag WITH tag'>'.
    PARSE VALUE text WITH '<'(tag)pre'style="'stl'"'rest
    IF pre = '>' THEN
    DO
        pre = ' '
        rest = '>'
    END
    IF POS( TRANSLATE( style_type ), TRANSLATE( stl ) ) > 0 THEN RETURN text
    IF LENGTH( mod_type ) > 0 THEN DO
        IF POS( mod_type, pre ) > 0 THEN DO
            PARSE VALUE pre WITH pre(mod_type)'="'mod_val'"'xtra
            rest = xtra||rest
        END
        ELSE IF POS( mod_type, rest ) > 0 THEN
        DO
            PARSE VALUE rest WITH xtra(mod_type)'="'mod_val'"'rest
            pre = pre||xtra
        END
/*        ELSE RETURN text*/
        IF LENGTH( style_setting ) = 0 THEN style_setting = mod_val
    END
RETURN '<'||tag||pre||'style="'||SPACE( stl||' '||style_type||' '||style_setting, 1 )||';"'||rest

adjust_chars: PROCEDURE EXPOSE __meter. link newln chars.
    old_trace = TRACE( 'O' )
    IF LENGTH( STRIP( ARG(1) ) ) = 0 THEN RETURN ARG(1)
    PARSE ARG alter, type_from, type_to
    IF \DATATYPE( type_from, 'W' ) THEN
        type_from = ( TRANSLATE( type_from ) = 'ASCII' ) + ( 2 * ( TRANSLATE( type_from ) = 'IPF' ) ) + ( 3 * ( TRANSLATE( type_from ) = 'HTML' ) )
    IF \DATATYPE( type_to, 'W' ) THEN
        type_to = ( TRANSLATE( type_to ) = 'ASCII' ) + ( 2 * ( TRANSLATE( type_to ) = 'IPF' ) ) + ( 3 * ( TRANSLATE( type_to ) = 'HTML' ) )
    tab = D2C(9)
    DO WHILE POS( '%', alter ) > 0
        PARSE VALUE alter WITH pre'%' +1 hex +2 post
        IF DATATYPE( hex, 'X' ) THEN
            alter = pre||X2C( hex )||post
        ELSE IF type_to = 2 THEN
            alter = pre||'&percent.'||hex||post
        ELSE IF type_to = 3 THEN
            alter = pre||'&percent;'||hex||post
        ELSE LEAVE
    END
    IF type_from < 0 | type_from > 3 | type_to < 1 | type_to > 3 THEN RETURN alter
    DO i = 1 TO chars.0
        PARSE VALUE chars.i WITH char.1(tab)char.2(tab)char.3
        IF LENGTH( char.1 ) = 0 THEN ITERATE i
        post = alter
        alter = ''
        DO j = 3 TO 2 BY -1
            DO WHILE POS( char.j, post ) > 0
                PARSE VALUE post WITH pre(char.j)post
                alter = alter||pre||char.1
            END
        END
        alter = alter||post
    END
    IF type_to > 1 THEN
    DO i = 1 TO chars.0
        PARSE VALUE chars.i WITH char.1(tab)char.2(tab)char.3
        IF LENGTH( char.type_to ) = 0 THEN ITERATE i
        post = alter
        alter = ''
        IF i = 2 & POS( '&amp.', post ) > 0 THEN
        DO
            DO WHILE POS( '&amp.', post ) > 0
                PARSE VALUE post WITH pre'&amp.'post
                DO WHILE POS( char.1, pre ) > 0
                    PARSE VALUE pre WITH before(char.1)pre
                    alter = alter||before||char.type_to
                END
                alter = alter||pre||'&amp.'
            END
            alter = alter||post
            ITERATE i
        END
        DO WHILE POS( char.1, post ) > 0
            PARSE VALUE post WITH pre(char.1)post
            alter = alter||pre||char.type_to
        END
        alter = alter||post
    END
    CALL TRACE old_trace
RETURN alter

chk_tags_2_parse: PROCEDURE EXPOSE __meter. link newln tags. input. output. chars. page_counter lnk_counter styles. active_tag colors.
    tab = D2C(9)
    DO i = 1 TO input.0
        CALL rxOut 0, 'Pass: '||FORMAT( ARG(1)/ARG(2),, 0 )||' ( of '||FORMAT( 1/ARG(2),, 0 )||' )'
        CALL rxWorking 1, input.i
        CALL RxProgress 2, ARG(1) + i / input.0 * ARG(2), ARG(1) + ARG(2), 1
        IF POS( '<', input.i ) = 0 THEN
            output.i = adjust_chars( input.i,, 'IPF' )
        ELSE DO
            input.i = adjust_chars( input.i )
            PARSE UPPER VALUE input.i WITH .'<'tag' '.
            PARSE VALUE tag WITH tag'>'.
            DO j = 1 TO tags.0
                PARSE VALUE tags.j WITH tgt.1(tab)tgt.2(tab)from.1(tab)from.2(tab)action
                DO k = 1 TO 2
                    IF LENGTH( from.k ) > 0 & TRANSLATE( from.k ) = tag THEN
                    DO
                        IF LENGTH( action ) = 1 THEN
                            output.i = output.i||tgt.k
                        ELSE
                            output.i = output.i||newln||tgt.k
                        LEAVE j
                    END
                END
            END
            IF j <= tags.0 & LENGTH( action ) > 0 THEN
                INTERPRET action
        END
    END
    output.0 = input.0
RETURN 0

parse_next_tag: PROCEDURE EXPOSE __meter. link newln counter input. output. tag. chr.
    PARSE ARG xmlString, tot_len, level, id, start, part
    IF \DATATYPE( input.0, 'W' ) THEN input.0 = 0
    IF \DATATYPE( level, 'W' ) THEN level = 0
    IF \DATATYPE( id, 'W' ) THEN id = 1
    DO WHILE POS( '<', xmlString ) > 0
        PARSE VALUE xmlString WITH pre'<'tag' 'xmlString
        IF LENGTH( STRIP( TRANSLATE( pre,, newln ) ) ) > 0 THEN
        DO
            count = input.0 + 1
            output.count = ''
            input.count = pre
            input.count.lvl = level
            id = id + 1
            input.count.0 = id
            input.0 = count
        END

        xmlString = '<'||tag||' '||xmlString
        IF POS( '>', tag ) > 0 THEN
            PARSE VALUE tag WITH tag'>'.
        tag = SPACE( TRANSLATE( tag,, newln ) )
        CALL rxOut 0, 'Pass: '||FORMAT( start/part,, 0 )||' ( of '||FORMAT( 1/part,, 0 )||' )'
        CALL RxWorking 1, tag
        CALL RxProgress 2, start + ( tot_len - LENGTH( xmlString ) ) / tot_len / part, start + part, 1
        PARSE VALUE xmlString WITH '<'start_tag'>'xml
        chk_xml = STRIP( STRIP( xml, 'T', D2C(10) ), 'T', D2C(13) )
        count = input.0
        IF input.count = '<td>' & start_tag = 'br' THEN
        DO
            count = input.0 + 1
            input.count = ''
        END
        ELSE DO
            count = input.0 + 1
            output.count = ''
            input.count = SPACE( TRANSLATE( '<'||start_tag||'>',, newln ) )
            input.count.lvl = level
            id = id + 1
            input.count.0 = id
            input.0 = count
        END
        end_tag = '</'||tag||'>'
        IF POS( end_tag, xmlString ) > 0 THEN DO
            xml = parse_xml_tags( xmlString, tag )
            chk_xml = STRIP( STRIP( STRIP( xml , 'L', D2C(13) ),, D2C(10) ), 'T', D2C(13) )
            level = level + 1
           IF LENGTH( chk_xml ) > 0 & POS( '<', chk_xml ) > 0 THEN
              CALL parse_next_tag chk_xml, tot_len, level, id, start, part
           ELSE IF LENGTH( chk_xml ) > 0  THEN DO
              count = input.0 + 1
              output.count = ''
              input.count = chk_xml
              input.count.lvl = level
              id = id + 1
              input.count.0 = id
              input.0 = count
           END
        END
        IF POS( end_tag, xmlString, LENGTH( '<'||start_tag||'>'||xml ) + 1 ) = 0 THEN
        DO
           end_tag = ''
           xml = ''
        END
        xmlString = SUBSTR( xmlString, LENGTH( '<'||start_tag||'>'||xml||end_tag ) + 1 )
        IF end_tag <> '' THEN DO
            level = level - 1
            count = input.0 + 1
            output.count = ''
            input.count = SPACE( TRANSLATE( end_tag,, newln ) )
            input.count.lvl = level
            id = id + 1
            input.count.0 = id
            input.0 = count
        END
    END
    IF LENGTH( STRIP( TRANSLATE( xmlString,, newln ) ) ) > 0 THEN
    DO
        count = input.0 + 1
        output.count = ''
        input.count = xmlString
        input.count.lvl = level
        id = id + 1
        input.count.0 = id
        input.0 = count
    END
RETURN ''

Usage:
    PARSE SOURCE . . this
    this = FILESPEC( 'N', this )
    SAY 'Usage:'
    SAY '      '||this||' input_file.html'
RETURN 1

parse_xml_tags: PROCEDURE /* xmlString, tag <<, tag_value>, selection> */
    prev_trace = TRACE( 'O' )
    PARSE UPPER ARG xmlString, tag, val, selection
    start_tag = '<'||tag
    INTERPRET "PARSE VALUE xmlString WITH pre'"||start_tag||val||"'extra'>'post"
    SELECT
        WHEN selection = 1 THEN /* tag */
        DO
            IF LENGTH( extra ) > 0 THEN
            DO
                PARSE VALUE SUBSTR( ARG(1), ABS( LENGTH( xmlString ) - LENGTH( post ) + 1 ) ) WITH pre'<'.
                RETURN pre
            END
        END
        WHEN selection = 2 THEN /* tag value */
            RETURN SUBSTR( ARG(1), ABS( LENGTH( xmlString ) - LENGTH( post ) - LENGTH( extra ) ), LENGTH( extra ) - ( RIGHT( extra, 1 ) = '/' ) )
        WHEN selection = 3 THEN /* tag value */
        DO
            tag_val = SUBSTR( ARG(1), ABS( LENGTH( xmlString ) - LENGTH( post ) - LENGTH( extra ) - LENGTH( val ) ), LENGTH( extra ) - ( RIGHT( extra, 1 ) = '/' ) + ( ( LENGTH( extra ) > 0 ) * LENGTH( val ) ) )
            INTERPRET "PARSE UPPER VALUE tag_val WITH re_val'"||val||'="'||"'ret_val'"||'"'||"'."
            IF LENGTH( STRIP( ret_val ) ) > 0 THEN RETURN ret_val
            IF LENGTH( STRIP( re_val ) ) > 0 THEN RETURN re_val
            INTERPRET "PARSE UPPER VALUE tag_val WITH re_val'"||val||"='ret_val ."
            IF LENGTH( STRIP( ret_val ) ) > 0 THEN RETURN ret_val
            IF LENGTH( STRIP( re_val ) ) > 0 THEN RETURN re_val
            INTERPRET "PARSE VALUE xmlString WITH pre'"||start_tag||"'ret_val'"||val||">'post"
            tag_end = LENGTH( pre ) + LENGTH( start_tag ) + LENGTH( ret_val ) + LENGTH( val ) + 2
            tag_val = REVERSE( SUBSTR( xmlString, 1, tag_end ) )
            INTERPRET "PARSE VALUE tag_val WITH '>"||REVERSE( val )||"'ret_val'"||REVERSE( start_tag )||"'post"
            IF LENGTH( STRIP( ret_val ) ) > 0 THEN
                RETURN SUBSTR( ARG(1), LENGTH( post ) + LENGTH( start_tag ) + 2, LENGTH( ret_val ) )
            ELSE RETURN '' /* What 2 do else?! */
        END
        WHEN selection = 4 THEN /* tag name value */
        DO
            INTERPRET "PARSE VALUE xmlString WITH pre'"||start_tag||" NAME="||'"'||STRIP( val,, '"' )||'"'||" VALUE="||'"'||"'extra'"||'"'||" />'."
            IF LENGTH( STRIP( extra ) ) = 0 THEN
                INTERPRET "PARSE VALUE xmlString WITH pre'"||start_tag||" NAME="||STRIP( val,, '"' )||" VALUE="||"'extra'"||"/'."
            IF LENGTH( STRIP( extra ) ) = 0 THEN RETURN ''
            start_pos = POS( extra, TRANSLATE( xmlString ) )
            IF start_pos > 0 THEN RETURN SUBSTR( ARG(1), start_pos, LENGTH( extra ) )
            ELSE RETURN ''
        END
        WHEN selection = 5 THEN /* tag value */
        DO
            INTERPRET "PARSE VALUE xmlString WITH .'"||start_tag||"'extra'>'.;PARSE VALUE extra WITH .'"||val||'="'||"'ret_val'"||'"'||"'."
            IF LENGTH( ret_val ) > 0 THEN RETURN SUBSTR( ARG(1), POS( ret_val, xmlString ), LENGTH( ret_val ) )
            ELSE RETURN ''
        END
        WHEN RIGHT( extra, 1 ) <> '/' THEN
        DO
            IF LENGTH( post ) = 0 THEN RETURN ''
            end_tag = '</'||tag||'>'
            next = LENGTH( xmlString ) - LENGTH( post ) + 1
            open_tag = next
            IF POS( end_tag, post ) = 0 THEN
            DO
                next = POS( '<', ARG(1), LENGTH( start_tag ) )
                IF next > 0 THEN
                DO
                    tag_len = LENGTH( pre||start_tag||val||extra||'>' ) + 1
                    RETURN SUBSTR( ARG(1), tag_len, next - tag_len )
                END
                ELSE RETURN ARG(1)
            END
            INTERPRET "PARSE VALUE post WITH pre'"||end_tag||"'post"
            close_tag = LENGTH( xmlString ) - LENGTH( post ) - LENGTH( end_tag ) + 1
            count = COUNTSTR( start_tag||'>', xmlString, open_tag, close_tag ) + COUNTSTR( start_tag||' ', xmlString, open_tag, close_tag )
            DO WHILE count > 0
                next = close_tag + LENGTH( end_tag )
                close_tag = POS( end_tag, xmlString, next )
                IF close_tag = 0 THEN
                    close_tag = LENGTH( xmlString ) /* Incomplete xml-string */
                IF close_tag > next THEN
                    post = SUBSTR( xmlString, next, close_tag - next )
                ELSE LEAVE

                INTERPRET "PARSE VALUE post WITH .'"||start_tag||"'extra'/>'post"
                DO WHILE LENGTH( post ) > 0
                    next = close_tag - LENGTH( post )
                    INTERPRET "PARSE VALUE post WITH .'"||start_tag||"'extra'/>'post"
                END
                count = COUNTSTR( start_tag||'>', xmlString, open_tag, close_tag ) + COUNTSTR( start_tag||' ', xmlString, open_tag, close_tag ) - 1
            END
            IF close_tag = 0 THEN close_tag = LENGTH( ARG(1) ) - open_tag
            ELSE close_tag = close_tag - open_tag
            IF close_tag = 0 THEN RETURN ''
            RETURN SUBSTR( ARG(1), open_tag, MAX( 1, close_tag ) )
        END
        OTHERWISE NOP
    END
    CALL TRACE prev_trace
RETURN ''

/* Count the number of occurances of needle in haystack from start pos to end pos (whole string by default) */
COUNTSTR: PROCEDURE /* needle, haystack< <, startpos>, endpos> */
    IF TRACE() = '?I' THEN CALL TRACE 'O'
    IF ARG() < 2 THEN RETURN -1
    IF DATATYPE( ARG(3), 'W' ) THEN
        next = ARG(3)
    ELSE
        next = 1
    needle = ARG(1)
    haystack = ARG(2)
    IF DATATYPE( ARG(4), 'W' ) THEN
        haystack = SUBSTR( haystack, next, ABS( ARG(4) - next ) )
    next = 1
    count = 0
    DO WHILE next > 0
        next = POS( needle, haystack, next )
        IF next > 0 THEN DO
            next = next + LENGTH( needle )
            count = count + 1
        END
    END
RETURN count

GetURLComponents: PROCEDURE EXPOSE host.
/* URL syntax consists of these components:          */
/* <scheme>://<location>:<port>/<path><params>?<query>#<target> */

    PARSE VALUE ARG( 1 ) WITH host.scheme '://' host.location '/' host.path
    host.location = strip( host.location, 'L', '/' ) /* in case of scheme like "file:///" */

    IF host.location = '' THEN
        PARSE VALUE ARG( 1 ) WITH host.location '/' host.path

    PARSE VALUE host.location WITH host.location ':' host.port
    IF host.port = '' THEN
        IF TRANSLATE( host.scheme ) = 'HTTP' then
            host.port = 80
        ELSE IF TRANSLATE( host.scheme ) = 'FTP' then
            host.port = 21

    PARSE VALUE host.path WITH host.file '' host.params '?' host.query '#' host.target

    IF host.params = '' THEN
        PARSE VALUE host.path WITH host.file '?' host.query '#' host.target

    IF host.query = '' THEN
        PARSE VALUE host.path WITH host.file '#' host.target

    host.family = "AF_INET"

RETURN

GetURLComponent: PROCEDURE EXPOSE host.

    IF LENGTH( ARG(1) ) > 0 THEN
        CALL GetURLComponents ARG( 1 )

    SELECT
        WHEN TRANSLATE( ARG( 2 ) ) = 'S' THEN
            RETURN host.scheme
        WHEN TRANSLATE( ARG( 2 ) ) = 'H' THEN
            RETURN host.location||':'||host.port
        WHEN TRANSLATE( ARG( 2 ) ) = 'N' THEN
            RETURN host.location
        WHEN TRANSLATE( ARG( 2 ) ) = 'L' THEN
            RETURN host.port
        WHEN TRANSLATE( ARG( 2 ) ) = 'P' THEN
            RETURN host.param
        WHEN TRANSLATE( ARG( 2 ) ) = 'Q' THEN
            RETURN host.query
        WHEN TRANSLATE( ARG( 2 ) ) = 'T' THEN
            RETURN host.target
        WHEN TRANSLATE( ARG( 2 ) ) = 'F' THEN
            RETURN FILESPEC( 'N', host.path )
        OTHERWISE
            RETURN host.path
    END

/* print an error message */
error: PROCEDURE
    CALL LINEOUT 'STDERR', ARG(1)
    EXIT

/* ------------------------------------------------------------------ */
/* function: Check if a file exists                                   */
/*                                                                    */
/* call:     FileExist fileToTest                                     */
/*                                                                    */
/* where:    fileToTest - name of the file to test                    */
/*                                                                    */
/* returns:  -2 - invalid parameter                                   */
/*           -1 - cannot detect (e.g. the drive is not ready)         */
/*            0 - neither a file, a device nor a directory with this  */
/*                name exist                                          */
/*            1 - the file exist                                      */
/*            2 - a device driver with the name exists                */
/*            3 - a directory with the name exists                    */
/*                                                                    */
FileExist: PROCEDURE
  parse arg fileName                                         /* v2.90 */

                        /* install temporary error handler            */
  SIGNAL ON NOTREADY NAME FileExistError
  SIGNAL ON FAILURE  NAME FileExistError
  SIGNAL ON ERROR    NAME FileExistError

  thisRC = -2           /* rc = -2 ->> invalid parameter              */

                        /* check the parameter                        */
  if strip( fileName ) <> "" then
  do
    thisRC = -1         /* rc = -1 ->> cannot detect the result       */

                        /* check if the drive with the file is ready  */
    call stream fileName
                        /* turn of some error handling so we can      */
                        /* determine if the given name is the name of */
                        /* a device (for example "LPT1")              */
    SIGNAL OFF NOTREADY

    if stream( fileName, "c", "QUERY EXISTS" ) <> "" then
    do
                        /* seems that the file exists -- check if     */
                        /* it is a device                             */
      if stream( fileName, "c", "QUERY DATETIME" ) == "" then
        thisRC = 2      /* rc = 2 ->> this is a device name           */
      else
        thisRC = 1      /* rc = 1 ->> this is a file name             */
    end /* if stream( ... */
    else
    do
                        /* seems that the file does not exist --      */
                        /* check if a directory with the name for the */
                        /* file exist                                 */

                        /* save the current directory of the current  */
                        /* drive                                      */
      thisDir = directory()
                        /* save the current directory of the drive    */
                        /* with the file to check                     */
      tempDir = directory( fileSpec( "Drive", fileName ) )

      if directory( fileName ) <> "" then
        thisRC = 3      /* rc = 3 ->> a dir with this name exists     */
      else
        thisRC = 0      /* rc = 0 ->> neither a file, a device nor a  */
                        /*            dir with this name exists       */

                        /* restore the current directory of the drive */
                        /* with the directory to check                */
      call directory tempDir
                        /* restore the current directory of the       */
                        /* current drive                              */
      call directory thisDir
    end /* else */
  end /* if strip( fileName ) <> "" then */

FileExistError:

RETURN thisRC

/* display_on_row, part  */
rxProgress: PROCEDURE EXPOSE  __meter.
    IF TRACE() = '?I' THEN RETURN 0
    row = ARG(1)
    IF SYMBOL( "__meter.row.t_stamp" ) = "VAR" THEN
        IF __meter.row.t_stamp + 1 > TIME( 'S' ) THEN RETURN 0
    chr = '۲ '
    IF SYMBOL( "__meter.width" ) <> "VAR" THEN
        PARSE VALUE SysTextScreenSize() WITH __meter.height __meter.width
    ELSE IF \DATATYPE( __meter.width, 'W' ) THEN
        PARSE VALUE SysTextScreenSize() WITH __meter.height __meter.width
    DO i = 2 TO MIN( ARG(), LENGTH( chr ) )
        j = i - 1
        IF LENGTH( ARG(i) ) = 0 THEN progress.j = 1
        ELSE IF \DATATYPE( ARG(i), 'N' ) THEN RETURN 0
        ELSE progress.j = FORMAT( ARG(i),,, 0 )
        IF LENGTH( FORMAT( 100 * progress.j,, 0, 0 ) ) > 3 THEN
            RETURN 0
    END
    progress.0 = j
    processed.0 = 0
    output = ''
    CALL SysStemSort 'progress'
    DO i = 1 TO MIN( progress.0, LENGTH( chr ) )
        j = i - 1
        processed.i = FORMAT( MIN( ( __meter.width - 4 ) * progress.i , __meter.width - 4 ),, 0, 0 )
        progress.i = FORMAT( 100 * progress.i, 3, 0, 0 )

        IF processed.i > processed.j THEN
            output = output||COPIES( SUBSTR( chr, i, 1 ), processed.i - processed.j )
    END
    i = i - 1
    output = output||COPIES( SUBSTR( chr, LENGTH( chr ), 1 ), __meter.width - 4 - processed.i )
    CALL rxOut ARG(1), output||RIGHT( progress.1||'%', 4 )
RETURN 0

/* display_on_row, text  */
rxWorking: PROCEDURE EXPOSE __meter.
    IF TRACE() = '?I' THEN RETURN 0
    PARSE ARG row, txt
    IF SYMBOL( "__meter."||row||".t_stamp" ) <> "VAR" THEN
        __meter.row.t_stamp = TIME( 'S' )
    ELSE IF \DATATYPE( __meter.row.t_stamp , 'N' ) THEN
        __meter.row.t_stamp = TIME( 'S' )
    IF __meter.row.t_stamp + 1 < TIME( 'S' ) THEN
    DO
        SELECT
            WHEN __meter.counter = 1 THEN
                CALL rxOut row, '/ '||txt
            WHEN __meter.counter = 2 THEN
                CALL rxOut row, '- '||txt
            WHEN __meter.counter = 3  THEN
                CALL rxOut row, '\ '||txt
            OTHERWISE
            __meter.counter = 0
            CALL rxOut row, '| '||txt
        END
        __meter.counter = __meter.counter + 1
    END
RETURN 0

/* display_on_row, text  */
rxOut: PROCEDURE EXPOSE __meter.
    IF TRACE() = '?I' THEN RETURN 0
    PARSE ARG row, txt
    IF SYMBOL( "__meter.width" ) <> "VAR" THEN
        PARSE VALUE SysTextScreenSize() WITH __meter.height __meter.width
    IF SYMBOL( "__meter."||row||".t_stamp" ) <> "VAR" THEN
        __meter.row.t_stamp = TIME( 'S' )
    IF \DATATYPE( __meter.row.t_stamp, 'W' ) | \DATATYPE( __meter.width, 'W' ) THEN
        PARSE VALUE SysTextScreenSize() WITH __meter.height __meter.width
    isNum = DATATYPE( row, 'W' )
    IF isNum THEN
    DO
        IF DATATYPE( __meter.row.t_stamp, 'W' ) THEN
            IF __meter.row.t_stamp + 1 > TIME( 'S' ) THEN RETURN 0
        PARSE VALUE SysCurPos( row, 0 ) with prev_row prev_col
    END
            ELSE IF DATATYPE( __meter.row.t_stamp, 'W' ) THEN
                IF __meter.row.t_stamp + 1 > TIME( 'S' ) THEN RETURN 0
    CALL CHAROUT 'STDERR', LEFT( txt, MAX( __meter.width, MIN( LENGTH( txt ), __meter.width ) ) )
    IF isNum THEN
    DO
        CALL SysCurPos prev_row, prev_col
        __meter.row.t_stamp = TIME( 'S' )
    END
    ELSE
        __meter.row.t_stamp = TIME( 'S' )
RETURN 0

/* Replace one string (needle) with another (newneedle) in text (haystack) */
CHANGESTR: PROCEDURE /* needle, haystack <, newneedle<, caselss>> */
    IF TRACE() = '?I' THEN CALL TRACE 'O'
    PARSE ARG needle, haystack, newneedle, caselss
    new_haystack = ''
    IF caselss = '' THEN
    DO WHILE POS( needle, haystack ) > 0
        PARSE VALUE haystack WITH pre(needle)haystack
        new_haystack = new_haystack||pre||newneedle
    END
    ELSE DO
        needle = TRANSLATE( needle )
        strpos = POS( needle, TRANSLATE( haystack ) )
        DO WHILE strpos > 0
            IF strpos > 1 THEN
                pre = LEFT( haystack, strpos - 1 )
            ELSE pre = ''
            haystack = SUBSTR( haystack, strpos + LENGTH( needle ) )
            new_haystack = new_haystack||pre||newneedle
            strpos = POS( needle, TRANSLATE( haystack ) )
        END
    END
RETURN new_haystack||haystack