(**************************************************************************)
(*                                                                        *)
(*  Admin for Major Major mailing list manager                            *)
(*  Copyright (C) 2025   Peter Moylan                                     *)
(*                                                                        *)
(*  This program is free software: you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation, either version 3 of the License, or     *)
(*  (at your option) any later version.                                   *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful,       *)
(*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *)
(*  GNU General Public License for more details.                          *)
(*                                                                        *)
(*  You should have received a copy of the GNU General Public License     *)
(*  along with this program.  If not, see <http://www.gnu.org/licenses/>. *)
(*                                                                        *)
(*  To contact author:   http://www.pmoylan.org   peter@pmoylan.org       *)
(*                                                                        *)
(**************************************************************************)

IMPLEMENTATION MODULE DKIMpage;

        (****************************************************************)
        (*                                                              *)
        (*                   PM Setup for Major Major                   *)
        (*                   DKIM page of the notebook                  *)
        (*                                                              *)
        (*        Started:        25 February 2024                      *)
        (*        Last edited:    19 December 2025                      *)
        (*        Status:         OK                                    *)
        (*                                                              *)
        (****************************************************************)


FROM SYSTEM IMPORT CARD8, ADDRESS, CAST, ADR;

IMPORT OS2, OS2RTL, DID, CommonSettings, Strings;

FROM Languages IMPORT
    (* type *)  LangHandle,
    (* proc *)  StrToBuffer;

FROM RINIData IMPORT
    (* proc *)  OpenINIFile, CloseINIFile,
                INIPut, INIFetch, INIGetCard,
                INIGetString, INIPutString;

FROM PMMisc IMPORT
    (* proc *)  WinAdjustDlgItemSize;

FROM Names IMPORT
    (* type *)  FilenameString, DomainName;

FROM LowLevel IMPORT
    (* proc *)  Copy;

(**************************************************************************)

CONST Nul = CHR(0);
      DisableChecking = TRUE;

VAR
    INIFileName: FilenameString;
    OurPageHandle, notebookhandle: OS2.HWND;
    ChangeInProgress: BOOLEAN;
    DKIMSignEnable: BOOLEAN;
    OurPageID: CARDINAL;

(************************************************************************)
(*                    OPERATIONS ON DIALOGUE LABELS                     *)
(************************************************************************)

PROCEDURE AdjustItemSizes;

    BEGIN
        (* ChangeInProgress := TRUE; *)
        WinAdjustDlgItemSize (OurPageHandle, DID.DKIMSignEnable);
        WinAdjustDlgItemSize (OurPageHandle, DID.DKIMsha1);
        WinAdjustDlgItemSize (OurPageHandle, DID.DKIMsha256);
        WinAdjustDlgItemSize (OurPageHandle, DID.hsimple);
        WinAdjustDlgItemSize (OurPageHandle, DID.hrelaxed);
        WinAdjustDlgItemSize (OurPageHandle, DID.bsimple);
        WinAdjustDlgItemSize (OurPageHandle, DID.brelaxed);
        (* ChangeInProgress := FALSE; *)
    END AdjustItemSizes;

(************************************************************************)

PROCEDURE SetLanguage (lang: LangHandle);

    (* Relabels this page in the new language. *)

    VAR stringval: ARRAY [0..511] OF CHAR;

    BEGIN
        StrToBuffer (lang, "DKIM.tab", stringval);
        OS2.WinSendMsg (notebookhandle, OS2.BKM_SETTABTEXT,
                        CAST(ADDRESS,OurPageID), ADR(stringval));
        StrToBuffer (lang, "DKIM.Enable", stringval);
        OS2.WinSetDlgItemText (OurPageHandle, DID.DKIMSignEnable, stringval);
        OS2.WinSetDlgItemText (OurPageHandle, DID.DKIMsha1, "rsa-sha1");
        OS2.WinSetDlgItemText (OurPageHandle, DID.DKIMsha256, "rsa-sha256");
        StrToBuffer (lang, "DKIM.SigningDomain", stringval);
        OS2.WinSetDlgItemText (OurPageHandle, DID.SigningLabel, stringval);
        StrToBuffer (lang, "DKIM.Selector", stringval);
        OS2.WinSetDlgItemText (OurPageHandle, DID.DKIMSelectorLabel, stringval);
        StrToBuffer (lang, "DKIM.DNSTXT", stringval);
        OS2.WinSetDlgItemText (OurPageHandle, DID.DKIMDNSTXTlocLabel, stringval);
        (*
        StrToBuffer (lang, "DKIM.copy", stringval);
        OS2.WinSetDlgItemText (OurPageHandle, DID.DKIMcopy, stringval);
        *)
        StrToBuffer (lang, "DKIM.PrivKeyFile", stringval);
        OS2.WinSetDlgItemText (OurPageHandle, DID.DKIMPrivKeyFileLabel, stringval);
        StrToBuffer (lang, "DKIM.Headers", stringval);
        OS2.WinSetDlgItemText (OurPageHandle, DID.DKIMHeadersLabel, stringval);
        StrToBuffer (lang, "DKIM.Canonic", stringval);
        OS2.WinSetDlgItemText (OurPageHandle, DID.DKIMcanonic, stringval);
        StrToBuffer (lang, "DKIM.header", stringval);
        OS2.WinSetDlgItemText (OurPageHandle, DID.DKIMh, stringval);
        StrToBuffer (lang, "DKIM.body", stringval);
        OS2.WinSetDlgItemText (OurPageHandle, DID.DKIMb, stringval);
        StrToBuffer (lang, "DKIM.simple", stringval);
        OS2.WinSetDlgItemText (OurPageHandle, DID.hsimple, stringval);
        OS2.WinSetDlgItemText (OurPageHandle, DID.bsimple, stringval);
        StrToBuffer (lang, "DKIM.relaxed", stringval);
        OS2.WinSetDlgItemText (OurPageHandle, DID.hrelaxed, stringval);
        OS2.WinSetDlgItemText (OurPageHandle, DID.brelaxed, stringval);

        AdjustItemSizes;

    END SetLanguage;

(**************************************************************************)
(*                 SHOW/HIDE SOME DIALOGUE ELEMENTS                       *)
(**************************************************************************)

PROCEDURE ShowHide (hwnd: OS2.HWND);

    (* Alters some dialogue elements.  *)

    (**********************************************************************)

    PROCEDURE DoIt (ID: CARDINAL;  value: BOOLEAN);

        BEGIN
            IF value THEN
                OS2.WinShowWindow (OS2.WinWindowFromID (hwnd, ID), TRUE);
            ELSE
                OS2.WinShowWindow (OS2.WinWindowFromID (hwnd, ID), FALSE);
            END (*IF*);
        END DoIt;

    (**********************************************************************)

    BEGIN
        DoIt (DID.DKIMsha1, DKIMSignEnable);
        DoIt (DID.DKIMsha256, DKIMSignEnable);
        DoIt (DID.SigningLabel, DKIMSignEnable);
        DoIt (DID.DKIMSigning, DKIMSignEnable);
        DoIt (DID.DKIMSelectorLabel, DKIMSignEnable);
        DoIt (DID.DKIMSelector, DKIMSignEnable);
        DoIt (DID.DKIMDNSTXTlocLabel, DKIMSignEnable);
        DoIt (DID.DKIMDNSTXTloc, DKIMSignEnable);
        DoIt (DID.DKIMPrivKeyFileLabel, DKIMSignEnable);
        DoIt (DID.DKIMPrivKeyFile, DKIMSignEnable);
        DoIt (DID.DKIMHeadersLabel, DKIMSignEnable);
        DoIt (DID.DKIMHeaders, DKIMSignEnable);
        DoIt (DID.DKIMcanonic, DKIMSignEnable);
        DoIt (DID.DKIMh, DKIMSignEnable);
        DoIt (DID.DKIMb, DKIMSignEnable);
        DoIt (DID.hsimple, DKIMSignEnable);
        DoIt (DID.hrelaxed, DKIMSignEnable);
        DoIt (DID.bsimple, DKIMSignEnable);
        DoIt (DID.brelaxed, DKIMSignEnable);
    END ShowHide;

(**************************************************************************)

PROCEDURE SetTxtLoc (hwnd: OS2.HWND);

    (* Works out the subdomain where the public key will be published. *)

    VAR domain, result: ARRAY [0..511] OF CHAR;

    BEGIN
        OS2.WinQueryDlgItemText (OurPageHandle, DID.DKIMSelector, 512, result);
        Strings.Append ("._domainkey.", result);
        OS2.WinQueryDlgItemText (OurPageHandle, DID.DKIMSigning, 512, domain);
        Strings.Append (domain, result);
        Strings.Append ('.', result);
        OS2.WinSetDlgItemText (hwnd, DID.DKIMDNSTXTloc, result);
    END SetTxtLoc;

(**************************************************************************)
(*                   LOADING AND STORING INI DATA                         *)
(**************************************************************************)

PROCEDURE LoadValues (hwnd: OS2.HWND);

    (* Fills the dialogue elements on page 1 with data from the INI file, *)
    (* or loads default values if they're not in the INI file.            *)

    CONST L = 512;

    VAR Selector, Signing: DomainName;
        text: ARRAY [0..L-1] OF CHAR;
        DKIMsr: CARD8;  bool: BOOLEAN;

    BEGIN
        IF NOT OpenINIFile (INIFileName) THEN
            RETURN;
        END (*IF*);

        (* Enable DKIM signing. *)

        IF NOT INIFetch ("$SYS", "DKIMSignEnable", DKIMSignEnable) THEN
            DKIMSignEnable := FALSE;
        END (*IF*);
        OS2.WinSendDlgItemMsg (hwnd, DID.DKIMSignEnable, OS2.BM_SETCHECK,
                         OS2.MPFROMSHORT(ORD(DKIMSignEnable)), NIL);

        (* Algorithm. *)

        IF NOT INIFetch ("$SYS", "DKIMsha1", bool) THEN
            bool := FALSE;
        END (*IF*);
        IF bool THEN
            OS2.WinSendDlgItemMsg (hwnd, DID.DKIMsha1, OS2.BM_SETCHECK,
                             OS2.MPFROMSHORT(1), NIL);
        ELSE
            OS2.WinSendDlgItemMsg (hwnd, DID.DKIMsha256, OS2.BM_SETCHECK,
                             OS2.MPFROMSHORT(1), NIL);
        END (*IF*);

        (* Signing domain. *)

        IF (NOT INIGetString ("$SYS", "DKIMSigning", Signing))
                    OR (Signing[0] = Nul) THEN
            Signing := "";
        END (*IF*);
        OS2.WinSetDlgItemText (hwnd, DID.DKIMSigning, Signing);

        (* Selector *)

        IF NOT INIGetString ("$SYS", "DKIMSelector", Selector) THEN
            Selector := "default";
        END (*IF*);
        OS2.WinSetDlgItemText (hwnd, DID.DKIMSelector, Selector);

        (* Initial value of DNS/txt location. *)

        OS2.WinSetDlgItemText (hwnd, DID.DKIMDNSTXTloc, "");

        (* Private key file *)

        IF NOT INIGetString ("$SYS", "DKIMPrivKeyFile", text) THEN
            text := "keys\priv.key";
        END (*IF*);
        OS2.WinSetDlgItemText (hwnd, DID.DKIMPrivKeyFile, text);

        (* Headers *)

        IF NOT INIGetString ("$SYS", "DKIMHeaders", text) THEN
            text := "From:To:Subject:Date:Message-ID";
        END (*IF*);
        OS2.WinSetDlgItemText (hwnd, DID.DKIMHeaders, text);

        (* The simple/relaxed choice. *)

        IF NOT INIFetch ("$SYS", "DKIMsr", DKIMsr) THEN
            DKIMsr := 3;
        END (*IF*);
        (*    00 = s/s     01 = s/r    10 = r/s    11 = r/r  *)
        IF ODD(DKIMsr) THEN
            OS2.WinSendDlgItemMsg (hwnd, DID.brelaxed, OS2.BM_SETCHECK,
                         OS2.MPFROMSHORT(1), NIL);
        ELSE
            OS2.WinSendDlgItemMsg (hwnd, DID.bsimple, OS2.BM_SETCHECK,
                         OS2.MPFROMSHORT(1), NIL);
        END (*IF*);
        IF ODD(DKIMsr DIV 2) THEN
            OS2.WinSendDlgItemMsg (hwnd, DID.hrelaxed, OS2.BM_SETCHECK,
                         OS2.MPFROMSHORT(1), NIL);
        ELSE
            OS2.WinSendDlgItemMsg (hwnd, DID.hsimple, OS2.BM_SETCHECK,
                         OS2.MPFROMSHORT(1), NIL);
        END (*IF*);

        CloseINIFile;
        ShowHide(hwnd);

    END LoadValues;

(**************************************************************************)

PROCEDURE StoreData;

    (* Stores the values on this page back into the INI file. *)

    CONST L = 512;

    VAR Signing, Selector, text: ARRAY [0..L-1] OF CHAR;
        DKIMsr: CARD8;  bool: BOOLEAN;

    BEGIN
        IF NOT OpenINIFile (INIFileName) THEN
            RETURN;
        END (*IF*);

        (* Enable DKIM signing. *)

        INIPut ('$SYS', 'DKIMSignEnable', DKIMSignEnable);

        (* Algorithm. *)

        bool := OS2.LONGFROMMR (OS2.WinSendDlgItemMsg (OurPageHandle, DID.DKIMsha1,
                                    OS2.BM_QUERYCHECK, NIL, NIL)) <> 0;
        INIPut ('$SYS', 'DKIMsha1', bool);

        (* Signing domain. *)

        OS2.WinQueryDlgItemText (OurPageHandle, DID.DKIMSigning, L, Signing);
        INIPutString ("$SYS", "DKIMSigning", Signing);

        (* Selector *)

        OS2.WinQueryDlgItemText (OurPageHandle, DID.DKIMSelector, L, Selector);
        INIPutString ("$SYS", "DKIMSelector", Selector);

        (* Private key file *)

        OS2.WinQueryDlgItemText (OurPageHandle, DID.DKIMPrivKeyFile, L, text);
        INIPutString ("$SYS", "DKIMPrivKeyFile", text);

        (* Headers *)

        OS2.WinQueryDlgItemText (OurPageHandle, DID.DKIMHeaders, L, text);
        INIPutString ("$SYS", "DKIMHeaders", text);

        (* The simple/relaxed choice. *)

        IF OS2.LONGFROMMR (OS2.WinSendDlgItemMsg (OurPageHandle, DID.hrelaxed,
                                    OS2.BM_QUERYCHECK, NIL, NIL)) <> 0 THEN
           DKIMsr := 2;
        ELSE
           DKIMsr := 0;
        END (*IF*);
        IF OS2.LONGFROMMR (OS2.WinSendDlgItemMsg (OurPageHandle, DID.brelaxed,
                                    OS2.BM_QUERYCHECK, NIL, NIL)) <> 0 THEN
           INC (DKIMsr);
        END (*IF*);
        INIPut ("$SYS", "DKIMsr", DKIMsr);

        CloseINIFile;

    END StoreData;

(**************************************************************************)

(*
PROCEDURE CopyField (hwnd: OS2.HWND);

    (* Copies the DID.DKIMDNSTXTloc value to the clipboard.  *)
    (* Not working, because of invalid parameter in DosAllocSharedMem,  *)
    (* but I don't yet know why.                                        *)

    VAR text: ARRAY [0..511] OF CHAR;
        loc, p: ADDRESS;
        hab: OS2.HAB;
        rc: CARDINAL;

    BEGIN
        p := ADR(loc);
        rc := OS2.DosAllocSharedMem (p, "\SHAREMEM\name", 512, OS2.OBJ_GIVEABLE+OS2.PAG_WRITE);
        IF rc <> 0 THEN
            err := rc;
        END (*IF*);
        OS2.WinQueryDlgItemText (hwnd, DID.DKIMDNSTXTloc, 512, text);
        Copy (ADR(text), loc, 512);
        OS2.WinOpenClipbrd (hab);
        OS2.WinEmptyClipbrd (hab);
        OS2.WinSetClipbrdData (hab, CAST(CARDINAL,loc), OS2.CF_TEXT, OS2.CFI_POINTER);
        OS2.WinCloseClipbrd (hab);
    END CopyField;
*)

(**************************************************************************)

PROCEDURE ["SysCall"] DialogueProc(hwnd     : OS2.HWND
                     ;msg      : OS2.ULONG
                     ;mp1, mp2 : OS2.MPARAM): OS2.MRESULT;

    VAR NotificationCode, ItemID: CARDINAL;

    BEGIN
        IF msg = OS2.WM_INITDLG THEN
            OS2.WinSetWindowPos (hwnd, 0, 0, 0, 0, 0, OS2.SWP_MOVE);
            LoadValues (hwnd);
            RETURN NIL;
        ELSIF msg = OS2.WM_PRESPARAMCHANGED THEN

            IF ChangeInProgress THEN
                RETURN OS2.WinDefDlgProc(hwnd, msg, mp1, mp2);
            ELSE
                ChangeInProgress := TRUE;
                CommonSettings.UpdateFontFrom (hwnd, CommonSettings.MainNotebook);
                ChangeInProgress := FALSE;
                RETURN NIL;
            END (*IF*);
        ELSIF msg = OS2.WM_COMMAND THEN
            (* "Copy" is the only button command we should receive. *)
            (*CopyField (hwnd);*)
        ELSIF msg = OS2.WM_CONTROL THEN
            NotificationCode := OS2.ULONGFROMMP(mp1);
            ItemID := NotificationCode MOD 65536;
            NotificationCode := NotificationCode DIV 65536;
            IF (ItemID = DID.DKIMSelector) OR (ItemID = DID.DKIMSigning) THEN
                IF NotificationCode = OS2.EN_CHANGE THEN
                    SetTxtLoc (hwnd);
                    RETURN NIL;
                ELSE
                    RETURN OS2.WinDefDlgProc(hwnd, msg, mp1, mp2);
                END (*IF*)
            ELSIF ItemID = DID.DKIMSignEnable THEN
                DKIMSignEnable := OS2.LONGFROMMR (OS2.WinSendDlgItemMsg (OurPageHandle,
                                    DID.DKIMSignEnable, OS2.BM_QUERYCHECK, NIL, NIL)) <> 0;
                ShowHide(hwnd);
            END (*IF*);
            RETURN OS2.WinDefDlgProc(hwnd, msg, mp1, mp2);
        ELSE
            RETURN OS2.WinDefDlgProc(hwnd, msg, mp1, mp2);
        END (*CASE*);
    END DialogueProc;

(**************************************************************************)

PROCEDURE CreatePage (notebook: OS2.HWND;  VAR (*OUT*) PageID: CARDINAL): OS2.HWND;

    (* Creates DKIM page and adds it to the notebook. *)

    BEGIN
        notebookhandle := notebook;
        OurPageHandle := OS2.WinLoadDlg(notebook, notebook,
                       DialogueProc,        (* dialogue procedure *)
                       0,                   (* use resources in EXE *)
                       DID.DKIMpage,        (* dialogue ID *)
                       NIL);                (* creation parameters *)
        PageID := OS2.ULONGFROMMR (OS2.WinSendMsg (notebook, OS2.BKM_INSERTPAGE,
                         NIL, OS2.MPFROM2SHORT (OS2.BKA_MAJOR+OS2.BKA_AUTOPAGESIZE, OS2.BKA_LAST)));
        OurPageID := PageID;
        OS2.WinSendMsg (notebook, OS2.BKM_SETPAGEWINDOWHWND,
                        CAST(ADDRESS,PageID), CAST(ADDRESS,OurPageHandle));
        SetTxtLoc (OurPageHandle);
        RETURN OurPageHandle;
    END CreatePage;

(**************************************************************************)

PROCEDURE SetFont (VAR (*IN*) name: CommonSettings.FontName);

    (* Sets the font of the text on this page. *)

    CONST bufsize = CommonSettings.FontNameSize;

    BEGIN
        OS2.WinSetPresParam (OurPageHandle, OS2.PP_FONTNAMESIZE, bufsize, name);
        AdjustItemSizes;
    END SetFont;

(**************************************************************************)

PROCEDURE SetINIFileName (name: ARRAY OF CHAR);

    (* Sets the INI file name. *)

    BEGIN
        Strings.Assign (name, INIFileName);
    END SetINIFileName;

(**************************************************************************)

BEGIN
    ChangeInProgress := FALSE;
    OurPageHandle := OS2.NULLHANDLE;
END DKIMpage.

