(**************************************************************************)
(*                                                                        *)
(*  Utility for the Weasel mail server                                    *)
(*  Copyright (C) 2021   Peter Moylan                                     *)
(*                                                                        *)
(*  This program is free software: you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation, either version 3 of the License, or     *)
(*  (at your option) any later version.                                   *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful,       *)
(*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *)
(*  GNU General Public License for more details.                          *)
(*                                                                        *)
(*  You should have received a copy of the GNU General Public License     *)
(*  along with this program.  If not, see <http://www.gnu.org/licenses/>. *)
(*                                                                        *)
(*  To contact author:   http://www.pmoylan.org   peter@pmoylan.org       *)
(*                                                                        *)
(**************************************************************************)

MODULE CalcMaxUsers;

        (********************************************************)
        (*                                                      *)
        (*     Works out recommendations for the "max users"    *)
        (*               entries in Weasel Setup                *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            11 March 2021                   *)
        (*  Last edited:        18 March 2021                   *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (********************************************************)


IMPORT Strings, IOChan, TextIO;

FROM ProgramArgs IMPORT
    (* proc *)  ArgChan, IsArgPresent;

FROM STextIO IMPORT
    (* proc *)  WriteChar, WriteString, WriteLn;

FROM SysClock IMPORT
    (* type *)  DateTime;

FROM TimeConv IMPORT
    (* proc *)  pack;

FROM FileOps IMPORT
    (* const*)  NoSuchChannel,
    (* type *)  ChanId,
    (* proc *)  OpenOldFile, OpenNewFile, CloseFile, ReadLine,
                Exists, DeleteFile, MoveFile,
                FWriteChar, FWriteString, FWriteLn, FWriteLJCard;

FROM RealMath IMPORT
    (* proc *)  sqrt;

FROM SRealIO IMPORT
    (* proc *)  WriteReal;

FROM Names IMPORT
    (* type *)  FilenameString;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

(************************************************************************)

TYPE
    Charset = SET OF CHAR;

CONST
    testing = FALSE;
    Nul = CHR(0);
    Digits = Charset {'0'..'9'};

    (*testparams = "CSVFILES\test.log CSVFILES";*)
    testparams = "CSVFILES\test.log";

TYPE
    EventType = (start, end);
    SessionType = (ignore, msa, pop, smtp, smtpout);

    EventPtr = POINTER TO EventRecord;
    EventRecord =   RECORD
                        previous, next: EventPtr;
                        time: CARDINAL;
                        duration: CARDINAL;
                        type: EventType;
                        sessnum: CARDINAL;
                    END (*RECORD*);

VAR
    (* The event list holds starts and ends of sessions.  If we are     *)
    (* going to generate a CSV file then we need to hold all of these.  *)
    (* If not, we keep a start record only until the matching end       *)
    (* record; then we update CountData, below, and then delete both    *)
    (* records.                                                         *)

    EventList:  ARRAY SessionType OF
                    RECORD
                        head, tail: EventPtr;
                    END (*RECORD*);

    (* CountData.count is the total number of sessions, and the *)
    (* intervalsum field is the sum of session durations.       *)

    CountData:  ARRAY SessionType OF
                    RECORD
                        firsttime, lasttime: CARDINAL;
                        count: CARDINAL;
                        intervalsum: CARDINAL;
                    END (*RECORD*);

    LogFile, CSVdir: FilenameString;

    MakeCSV: BOOLEAN;

(************************************************************************)
(*                    MISCELLANEOUS CONVERSION ROUTINES                 *)
(************************************************************************)

PROCEDURE GetCard (stamp: ARRAY OF CHAR;  VAR (*INOUT*) k: CARDINAL): CARDINAL;

    (* Converts a cardinal starting at stamp[k], updates k.  *)

    VAR result: CARDINAL;  ch: CHAR;

    BEGIN
        result := 0;
        ch := stamp[k];
        WHILE ch IN Digits DO
            result := 10*result + ORD(ch) - ORD('0');
            INC (k);
            ch := stamp[k];
        END (*WHILE*);
        RETURN result;
    END GetCard;

(************************************************************************)

PROCEDURE TimeStampToSecs (stamp: ARRAY OF CHAR): CARDINAL;

    (* Convertes a timestamp to seconds since an arbitrary origin. *)

    VAR DT: DateTime;
        k, secs: CARDINAL;

    BEGIN
        k := 0;
        WITH DT DO
            year           := GetCard (stamp, k);  INC(k);
            month          := GetCard (stamp, k);  INC(k);
            day            := GetCard (stamp, k);  INC(k);
            hour           := GetCard (stamp, k);  INC(k);
            minute         := GetCard (stamp, k);  INC(k);
            second         := GetCard (stamp, k);
            fractions      := 0;
            zone           := 0;
            SummerTimeFlag := FALSE;
        END (*WITH*);
        pack (DT, secs);
        RETURN secs;
    END TimeStampToSecs;

(************************************************************************)

PROCEDURE GetNum (label: ARRAY OF CHAR): CARDINAL;

    (* Skips any nonnumeric characters at the beginning of label,   *)
    (* interprets what remains to a decimal number.                 *)

    VAR k: CARDINAL;

    BEGIN
        k := 0;
        WHILE (k <= HIGH(label)) AND NOT (label[k] IN Digits) DO
            INC (k);
        END (*WHILE*);
        IF k > HIGH(label) THEN
            RETURN 0;
        ELSE
            RETURN GetCard (label, k);
        END (*IF*);
    END GetNum;

(************************************************************************)

PROCEDURE WriteCard (N: CARDINAL);

    BEGIN
        IF N > 9 THEN
            WriteCard (N DIV 10);
            N := N MOD 10;
        END (*IF*);
        WriteChar (CHR(ORD('0')+N));
    END WriteCard;

(************************************************************************)
(*                        GET PROGRAM ARGUMENTS                         *)
(************************************************************************)

PROCEDURE GetParams;

    VAR args: IOChan.ChanId;
        pos: CARDINAL;  found: BOOLEAN;

    BEGIN
        CSVdir := "";  MakeCSV := FALSE;
        IF testing THEN
            LogFile := testparams;
        ELSE
            args := ArgChan();
            IF IsArgPresent() THEN
                TextIO.ReadString (args, LogFile);
            END (*IF*);
        END (*IF*);
        IF LogFile[0] = Nul THEN
            LogFile := "Weasel.log";
        ELSE
            Strings.FindNext (' ', LogFile, 0, found, pos);
            IF found THEN
                CSVdir := LogFile;
                Strings.Delete (CSVdir, 0, pos+1);
                LogFile[pos] := Nul;
                WHILE CSVdir[0] = ' ' DO
                    Strings.Delete (CSVdir, 0, 1);
                END (*WHILE*);
                MakeCSV := CSVdir[0] <> Nul;
                IF MakeCSV THEN
                    pos := Strings.Length(CSVdir)-1;
                    IF (CSVdir[pos] = '/') OR (CSVdir[pos] = '\') THEN
                        CSVdir[pos] := Nul;
                    END (*IF*);
                    Strings.Append ('\', CSVdir);
                END (*IF*);
            END (*IF*);
        END (*IF*);
    END GetParams;

(************************************************************************)
(*                       DUMP EVENTS TO CSV FILE                        *)
(************************************************************************)

PROCEDURE WriteCSVline (cid: ChanId;  p: EventPtr;  lineno: CARDINAL);

    BEGIN
        FWriteLJCard (cid, p^.time);  FWriteChar (cid, ',');
        FWriteLJCard (cid, p^.sessnum);  FWriteChar (cid, ',');
        IF p^.type = start THEN
            FWriteLJCard (cid, p^.duration);  FWriteChar (cid, ',');
            FWriteString (cid, '+1');  FWriteChar (cid, ',');
        ELSE
            FWriteString (cid, ',-1');  FWriteChar (cid, ',');
        END (*IF*);
        FWriteString (cid, "=E");  FWriteLJCard (cid, lineno-1);
        FWriteString (cid, "+D");  FWriteLJCard (cid, lineno);
        FWriteLn (cid);
    END WriteCSVline;

(************************************************************************)

PROCEDURE WriteToCSV (st: SessionType);

    CONST csvtemp = "LOGx.CSV";  baktemp = "LOGx.BAK";

    TYPE TT = ARRAY SessionType OF CHAR;

    CONST tag = TT{'x', 'M', 'P', 'S', 'O'};

    VAR cid: ChanId;  lineno: CARDINAL;
        p: EventPtr;
        csvname, bakname: FilenameString;

    BEGIN
        p := EventList[st].head;
        IF p = NIL THEN
            WriteString ("Event list ");  WriteChar (tag[st]);
            WriteString (" is empty");  WriteLn;
            RETURN;
        END (*IF*);
        csvname := csvtemp;
        bakname := baktemp;
        csvname[3] := tag[st];
        bakname[3] := tag[st];
        Strings.Insert (CSVdir, 0, csvname);
        Strings.Insert (CSVdir, 0, bakname);

        IF Exists (csvname) THEN
            IF Exists (bakname) THEN
                DeleteFile (bakname);
            END (*IF*);
            EVAL (MoveFile (csvname, bakname));
        END (*IF*);

        cid := OpenNewFile (csvname, FALSE);

        (* Write some headings.  *)

        CASE st OF
              msa:      FWriteString (cid, "Message submission");  FWriteLn (cid);
            | pop:      FWriteString (cid, "POP");  FWriteLn (cid);
            | smtp:     FWriteString (cid, "SMTP");  FWriteLn (cid);
            | smtpout:  FWriteString (cid, "MAIL OUT");  FWriteLn (cid);
        ELSE
            (* Write nothing *)
        END (*CASE*);
        FWriteString (cid, "time,session #,duration,start/end,in progress");
        FWriteLn (cid);

        lineno := 3;
        WHILE p <> NIL DO
            WriteCSVline (cid, p, lineno);
            p := p^.next;
            INC (lineno);
        END (*WHILE*);
        CloseFile (cid);
    END WriteToCSV;

(************************************************************************)
(*                           STRING MATCHING                            *)
(************************************************************************)

PROCEDURE HeadMatch (VAR (*IN*) str: ARRAY OF CHAR;  tomatch: ARRAY OF CHAR): BOOLEAN;

    (* Returns TRUE iff a leading substring of str matches tomatch. *)

    VAR k: CARDINAL;

    BEGIN
        k := 0;
        LOOP
            IF k >= LENGTH(tomatch) THEN
                RETURN TRUE;
            ELSIF k >= LENGTH(str) THEN
                RETURN FALSE;
            ELSIF str[k] <> tomatch[k] THEN
                RETURN FALSE;
            ELSE
                INC (k);
            END (*IF*);
        END (*LOOP*);
    END HeadMatch;

(************************************************************************)
(*                    OPERATIONS ON THE EVENT LIST                      *)
(************************************************************************)

PROCEDURE ClearGlobalData;

    (* Sets the initial state of the event list and the counters. *)

    VAR st: SessionType;

    BEGIN
        FOR st := MIN(SessionType) TO MAX(SessionType) DO
            EventList[st].head := NIL;
            EventList[st].tail := NIL;
            CountData[st].firsttime := MAX(CARDINAL);
            CountData[st].lasttime := 0;
            CountData[st].count := 0;
            CountData[st].intervalsum := 0;
        END (*FOR*);
    END ClearGlobalData;

(************************************************************************)

PROCEDURE PutEvent (time: CARDINAL;  type: EventType;
                            sessnum: CARDINAL;  sesstype: SessionType);

    (* Adds an event to the event list.  *)

    VAR p: EventPtr;

    BEGIN
        NEW (p);
        p^.time := time;
        p^.type := type;
        p^.sessnum := sessnum;
        p^.duration := MAX(CARDINAL);       (* to be filled in later *)
        p^.previous := EventList[sesstype].tail;
        IF EventList[sesstype].tail = NIL THEN
            EventList[sesstype].head := p;
        ELSE
            EventList[sesstype].tail^.next := p;
        END (*IF*);
        p^.next := NIL;
        EventList[sesstype].tail := p;
    END PutEvent;

(************************************************************************)

PROCEDURE MatchingStart (sesstype: SessionType;  sessnum: CARDINAL): EventPtr;

    (* Returns the start record for this session.  We have to search    *)
    (* backwards through the list, because session numbers can be       *)
    (* re-used.                                                         *)

    VAR p: EventPtr;

    BEGIN
        p := EventList[sesstype].tail;
        WHILE (p <> NIL) AND
                    ((p^.sessnum <> sessnum) OR (p^.type <> start)) DO
             p := p^.previous;
        END (*WHILE*);
        RETURN p;
    END MatchingStart;

(************************************************************************)

PROCEDURE DeleteEvent (st: SessionType;  VAR (*INOUT*) p: EventPtr);

    (* Removes p^ from its event list.  *)

    BEGIN
        IF p^.previous = NIL THEN
            EventList[st].head := p^.next;
        ELSE
            p^.previous^.next := p^.next;
        END (*IF*);
        IF p^.next = NIL THEN
            EventList[st].tail := p^.previous;
        ELSE
            p^.next^.previous := p^.previous;
        END (*IF*);
        DISPOSE (p);
    END DeleteEvent;

(************************************************************************)
(*                        SCANNING THE LOG FILE                         *)
(************************************************************************)

PROCEDURE IsStartLine (sesstype: SessionType;  message: ARRAY OF CHAR): BOOLEAN;

    (* Returns TRUE if message is a starting line for a session.    *)

    BEGIN
        CASE sesstype OF
              ignore:   RETURN FALSE;
            | msa, pop, smtp:
                        RETURN HeadMatch (message, "New client");
            | smtpout:  RETURN HeadMatch (message, "Attempting to");
            ELSE
                        RETURN FALSE;
        END (*CASE*);
    END IsStartLine;

(************************************************************************)

PROCEDURE IsTerminationLine (sesstype: SessionType;  message: ARRAY OF CHAR): BOOLEAN;

    (* Returns TRUE if message is a session termination line for this   *)
    (* kind of session.                                                 *)

    BEGIN
        CASE sesstype OF
              ignore:   RETURN FALSE;
            | pop:
                        RETURN HeadMatch (message, "End of session")
                            OR HeadMatch (message, "> -ERR Access denied")
                            OR HeadMatch (message, "> -ERR User limit")
                            OR HeadMatch (message, "> -ERR Temporar");
            | msa, smtp:
                        RETURN HeadMatch (message, "End of session")
                            OR HeadMatch (message, "> 421 Access denied")
                            OR HeadMatch (message, "> 421 User limit")
                            OR HeadMatch (message, "> 421 Temporar")
                            OR HeadMatch (message, "> 571 Connection refused")
                            OR HeadMatch (message, "> Out of memory")
                            OR HeadMatch (message, "Client rejected");
            | smtpout:  RETURN HeadMatch (message, "Delivered")
                            OR HeadMatch (message, "Failed");
        ELSE
                        RETURN FALSE;
        END (*CASE*);
    END IsTerminationLine;

(************************************************************************)

PROCEDURE CollectData;

    (* Gets the log file specification, extracts the relevant   *)
    (* information from it.                                     *)

    CONST CtrlZ = CHR(26);

    VAR logline, content: ARRAY [0..511] OF CHAR;
        timestamp: ARRAY [0..19] OF CHAR;
        label: ARRAY [0..7] OF CHAR;
        time, sessnum: CARDINAL;
        cid: ChanId;
        sesstype: SessionType;
        q: EventPtr;

    BEGIN
        (* Open the log file. *)

        cid := OpenOldFile (LogFile, FALSE, FALSE);
        IF cid = NoSuchChannel THEN
            WriteString ("Cannot open log file ");
            WriteString (LogFile);  WriteLn;
            RETURN;
        END (*IF*);

        (* Main data collection loop.  *)

        ClearGlobalData;
        ReadLine (cid, logline);
        WHILE logline[0] <> CtrlZ DO

            (* Decode the content of this line. *)

            Strings.Extract (logline, 0, 19, timestamp);
            time := TimeStampToSecs (timestamp);
            Strings.Extract (logline, 20, 7, label);
            sessnum := GetNum (label);
            Strings.Assign (logline, content);
            Strings.Delete (content, 0, 29);

            (* The session type depends on what sort of label it is. *)

            IF (label[0] = ' ') OR (label[0] = '*')
                  OR HeadMatch (label, 'Setup') OR HeadMatch (label, 'Sorter')
                            OR HeadMatch (label, 'Online') THEN
                sesstype := ignore;
            ELSIF HeadMatch (label, 'Send_') THEN
                (* SMTP OUTPUT RECORD *)
                sesstype := smtpout;
            ELSE
                CASE label[0] OF
                      'M':  sesstype := msa;
                    | 'P':  sesstype := pop;
                    | 'S':  sesstype := smtp;
                ELSE
                    sesstype := ignore;
                END (*CASE*);
            END (*IF*);

            (* Look for lines that start and end a session. *)

            IF sesstype = ignore THEN
                (* Do nothing - ignore this line *)
            ELSE
                IF IsStartLine (sesstype, content) THEN
                    PutEvent (time, start, sessnum, sesstype);
                ELSIF IsTerminationLine (sesstype, content) THEN

                    q := MatchingStart (sesstype, sessnum);

                    (* If no matching start record, ignore this event. *)

                    IF q <> NIL THEN
                        q^.duration := time - q^.time;
                        WITH CountData[sesstype] DO
                            INC (count);
                            INC (intervalsum, q^.duration);
                            IF q^.time < firsttime THEN
                                firsttime := q^.time;
                            END (*IF*);
                            IF time > lasttime THEN
                                lasttime := time;
                            END (*IF*);
                        END (*WITH*);
                        IF MakeCSV THEN
                            PutEvent (time, end, sessnum, sesstype);
                        ELSE
                            DeleteEvent (sesstype, q);
                        END (*IF*);
                    END (*IF*);

                END (*IF*);

            END (*IF*);

            ReadLine (cid, logline);

        END (*WHILE*);

        CloseFile (cid);

    END CollectData;

(************************************************************************)
(*                          THE OVERALL CALCULATION                     *)
(************************************************************************)

PROCEDURE DoTheCalcs;

    VAR st: SessionType;
        p, next: EventPtr;
        starttime, endtime, time: CARDINAL;
        lambda, oneonmu, mean: REAL;

    BEGIN
        WriteString ("Program to calculate recommended 'Max Users' values");
        WriteLn;
        CollectData;

        (* Remove sessions where we don't have the completion line. *)

        FOR st := MIN(SessionType) TO MAX(SessionType) DO
            p := EventList[st].head;
            WHILE p <> NIL DO
                next := p^.next;
                IF (p^.type = start) AND (p^.duration = MAX(CARDINAL)) THEN
                    DeleteEvent (st, p);
                END (*IF*);
                p := next;
            END (*WHILE*);
        END (*FOR*);

        (* Putting out the optional CSV files. *)

        IF MakeCSV THEN
            WriteString ("Creating CSV files");  WriteLn;
            FOR st := MIN(SessionType) TO MAX(SessionType) DO
                IF st <> ignore THEN
                    WriteToCSV (st);
                END (*IF*);
            END (*FOR*);
        END (*IF*);

        (* Calculate total time. *)

        starttime := MAX(CARDINAL);  endtime := 0;
        FOR st := MIN(SessionType) TO MAX(SessionType) DO
            IF CountData[st].firsttime < starttime THEN
                starttime := CountData[st].firsttime;
            END (*IF*);
            IF CountData[st].lasttime > endtime THEN
                endtime := CountData[st].lasttime;
            END (*IF*);
        END (*FOR*);
        time := endtime - starttime;
        WriteString ("Samples taken over ");  WriteCard (time);
        WriteString (" seconds");  WriteLn;

        (* Now for the queueing theory calculations. *)

        WriteLn;  WriteString ("SESSION STATISTICS");  WriteLn;
        FOR st := MIN(SessionType) TO MAX(SessionType) DO
            CASE st OF
                  msa:      WriteString ("MSA ");
                | pop:      WriteString ("POP ");
                | smtp:     WriteString ("SMTP ");
                | smtpout:  WriteString ("MAIL OUT ");
            ELSE
                (* Write nothing *)
            END (*CASE*);
            IF CountData[st].count = 0 THEN
                IF st <> ignore THEN
                    WriteString ("(no data)");
                END (*IF*);
            ELSE
                WriteCard (CountData[st].count);
                WriteString (" samples  Arrival rate ");
                lambda := FLOAT(CountData[st].count) / FLOAT(time);
                WriteReal (lambda, 7);
                WriteString ("/sec  ");
                WriteString ("Mean service time ");
                oneonmu := FLOAT (CountData[st].intervalsum) / FLOAT(time) + 0.5;
                WriteReal (oneonmu, 7);
                WriteString (" sec");
                WriteLn;
                mean := lambda*oneonmu;
                WriteString ("    Mean traffic level ");
                WriteReal (mean, 7);
                WriteString ("       Recommended max users: ");
                WriteCard (TRUNC(mean + 3.0*sqrt(mean) + 2.0));
            END (*IF*);
            WriteLn;
            WriteLn;

        END (*FOR*);

    END DoTheCalcs;

(************************************************************************)
(*                             MAIN PROGRAM                             *)
(************************************************************************)

BEGIN
    GetParams;
    DoTheCalcs;
END CalcMaxUsers.

